<?php
include 'header.php';

// ---------------------------------------------------------------------
// 1. Get stock (via id or symbol)
// ---------------------------------------------------------------------
$stock_id = (int)($_GET['id'] ?? 0);
$symbol   = strtoupper(trim($_GET['symbol'] ?? ''));

if ($stock_id <= 0 && $symbol === '') {
    die('Invalid request.');
}

// Fetch stock
if ($stock_id > 0) {
    $stmt = mysqli_prepare($link, "SELECT symbol, name, price, previous_close, volume, market_cap FROM stocks WHERE id = ?");
    mysqli_stmt_bind_param($stmt, "i", $stock_id);
} else {
    $stmt = mysqli_prepare($link, "SELECT id, name, price, previous_close, volume, market_cap FROM stocks WHERE symbol = ?");
    mysqli_stmt_bind_param($stmt, "s", $symbol);
}
mysqli_stmt_execute($stmt);
mysqli_stmt_bind_result($stmt, $db_symbol, $name, $price, $prev_close, $volume, $market_cap);
if (!mysqli_stmt_fetch($stmt)) {
    die('Stock not found.');
}
mysqli_stmt_close($stmt);

if ($stock_id <= 0) $stock_id = $db_symbol;
$symbol = $db_symbol;

// ---------------------------------------------------------------------
// 2. Calculate change
// ---------------------------------------------------------------------
$change     = $price - $prev_close;
$change_pct = $prev_close > 0 ? ($change / $prev_close) * 100 : 0;
$change_sign = $change >= 0 ? '+' : '';
$change_cls  = $change >= 0 ? 'text-green-600' : 'text-red-600';
$change_icon = $change >= 0 ? 'trending-up' : 'trending-down';

// ---------------------------------------------------------------------
// 3. Fetch user recent transactions
// ---------------------------------------------------------------------
$transactions = [];
$stmt = mysqli_prepare($link, "
    SELECT type, shares, price_per_share, total_amount, fee, created_at 
    FROM stock_transactions 
    WHERE user_id = ? AND stock_id = ? 
    ORDER BY created_at DESC 
    LIMIT 10
");
mysqli_stmt_bind_param($stmt, "ii", $user_id, $stock_id);
mysqli_stmt_execute($stmt);
$result = mysqli_stmt_get_result($stmt);
while ($row = mysqli_fetch_assoc($result)) {
    $transactions[] = $row;
}
mysqli_stmt_close($stmt);
?>

<!-- ================================================================ -->
<!-- ======================= MAIN LAYOUT ============================ -->
<!-- ================================================================ -->
<div class="flex-1 flex flex-col overflow-hidden lg:ml-72">
    <!-- Top Bar -->
    <header class="bg-white border-b border-gray-200">
        <div class="flex items-center justify-between px-6 py-3">
            <div class="flex items-center">
                <button onclick="toggleSidebar()" class="lg:hidden text-gray-500 hover:text-gray-700 p-1.5 rounded-md hover:bg-gray-100 mr-3">
                    <i data-lucide="menu" class="w-5 h-5"></i>
                </button>
                <div class="font-light text-lg text-black">
                    <?= htmlspecialchars($symbol) ?> - <?= htmlspecialchars($name) ?>
                </div>
            </div>
            <!--<div class="flex items-center space-x-3">-->
            <!--    <div class="relative">-->
            <!--        <button onclick="toggleNotifications()" class="relative p-2 text-gray-500 hover:text-gray-700 rounded-md hover:bg-gray-100">-->
            <!--            <i data-lucide="bell" class="w-5 h-5"></i>-->
            <!--            <span id="notification-badge" class="absolute -top-1 -right-1 bg-red-500 text-white text-xs rounded-full h-4 w-4 flex items-center justify-center hidden">0</span>-->
            <!--        </button>-->
            <!--    </div>-->
            <!--</div>-->
        </div>
    </header>

    <!-- Page Content -->
    <main class="flex-1 overflow-y-auto p-6 bg-white pb-20 lg:pb-6">
        <div class="max-w-7xl mx-auto">
            <!-- Header -->
            <div class="bg-gradient-to-br from-black via-gray-900 to-gray-800 rounded-2xl p-6 mb-6 text-white relative overflow-hidden">
                <div class="absolute inset-0 opacity-10">
                    <div class="absolute top-0 right-0 w-48 h-48 bg-white rounded-full -translate-y-24 translate-x-24"></div>
                    <div class="absolute bottom-0 left-0 w-24 h-24 bg-white rounded-full translate-y-12 -translate-x-12"></div>
                </div>
                <div class="relative z-10">
                    <div class="flex flex-col lg:flex-row lg:items-center lg:justify-between">
                        <div class="mb-4 lg:mb-0 lg:flex-1">
                            <h1 class="text-xl font-light mb-1"><?= htmlspecialchars($symbol) ?></h1>
                            <p class="text-gray-300 text-sm"><?= htmlspecialchars($name) ?></p>
                            <p class="text-xs text-gray-400 mt-1">Technology • Technology</p>
                        </div>
                        <div class="bg-white bg-opacity-15 backdrop-blur-xl rounded-xl p-4 border border-white border-opacity-20 shadow-xl lg:w-80">
                            <div class="flex items-center justify-between mb-3">
                                <div>
                                    <p class="text-xs text-gray-300 mb-1">Current Price</p>
                                    <p class="text-lg font-light">$<?= number_format($price, 2) ?></p>
                                </div>
                                <div class="w-10 h-10 flex items-center justify-center">
                                    <i data-lucide="bar-chart-3" class="w-5 h-5 text-white"></i>
                                </div>
                            </div>
                            <div class="flex space-x-4 text-xs">
                                <div class="flex-1 text-center">
                                    <p class="text-gray-300">Change</p>
                                    <p class="<?= $change_cls ?> font-medium">
                                        <?= $change_sign ?>$<?= number_format(abs($change), 2) ?>
                                    </p>
                                </div>
                                <div class="flex-1 text-center">
                                    <p class="text-gray-300">Volume</p>
                                    <p class="text-white font-medium"><?= number_format($volume / 1_000_000, 1) ?>M</p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Stock Overview -->
            <div class="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
                <div class="bg-white rounded-xl p-4 shadow-sm border border-gray-100">
                    <div class="flex items-center justify-between mb-3">
                        <div>
                            <p class="text-xs text-gray-500 mb-1">Current Price</p>
                            <p class="text-lg font-light text-black">$<?= number_format($price, 2) ?></p>
                        </div>
                        <div class="w-8 h-8 flex items-center justify-center">
                            <i data-lucide="dollar-sign" class="w-4 h-4 text-blue-600"></i>
                        </div>
                    </div>
                </div>
                <div class="bg-white rounded-xl p-4 shadow-sm border border-gray-100">
                    <div class="flex items-center justify-between mb-3">
                        <div>
                            <p class="text-xs text-gray-500 mb-1">Change</p>
                            <p class="text-lg font-light <?= $change_cls ?>">
                                <?= $change_sign ?>$<?= number_format(abs($change), 2) ?>
                            </p>
                            <p class="text-xs <?= $change_cls ?>">
                                <?= $change_sign ?><?= number_format($change_pct, 2) ?>%
                            </p>
                        </div>
                        <div class="w-8 h-8 flex items-center justify-center">
                            <i data-lucide="<?= $change_icon ?>" class="w-4 h-4 <?= $change_cls ?>"></i>
                        </div>
                    </div>
                </div>
                <div class="bg-white rounded-xl p-4 shadow-sm border border-gray-100">
                    <div class="flex items-center justify-between mb-3">
                        <div>
                            <p class="text-xs text-gray-500 mb-1">Volume</p>
                            <p class="text-lg font-light text-black"><?= number_format($volume / 1_000_000, 1) ?>M</p>
                        </div>
                        <div class="w-8 h-8 flex items-center justify-center">
                            <i data-lucide="activity" class="w-4 h-4 text-green-600"></i>
                        </div>
                    </div>
                </div>
                <div class="bg-white rounded-xl p-4 shadow-sm border border-gray-100">
                    <div class="flex items-center justify-between mb-3">
                        <div>
                            <p class="text-xs text-gray-500 mb-1">Market Cap</p>
                            <p class="text-lg font-light text-black">$<?= number_format($market_cap / 1_000_000, 1) ?>M</p>
                        </div>
                        <div class="w-8 h-8 flex items-center justify-center">
                            <i data-lucide="pie-chart" class="w-4 h-4 text-purple-600"></i>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Main Content -->
            <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
                <!-- Stock Information -->
                <div class="lg:col-span-2 space-y-6">
                    <!-- Price Chart -->
                    <div class="bg-white rounded-xl p-6 shadow-sm border border-gray-100">
                        <div class="flex items-center justify-between mb-4">
                            <div>
                                <h3 class="text-lg font-light text-black mb-1">Price Chart</h3>
                                <p class="text-xs text-gray-500">Stock performance over time</p>
                            </div>
                            <div class="flex space-x-2">
                                <button data-period="1d" class="px-3 py-1 text-xs font-medium text-gray-600 hover:text-black transition-colors duration-200">1D</button>
                                <button data-period="1w" class="px-3 py-1 text-xs font-medium text-gray-600 hover:text-black transition-colors duration-200">1W</button>
                                <button data-period="1m" class="px-3 py-1 text-xs font-medium text-black bg-gray-100 rounded-lg">1M</button>
                                <button data-period="1y" class="px-3 py-1 text-xs font-medium text-gray-600 hover:text-black transition-colors duration-200">1Y</button>
                            </div>
                        </div>
                        <div class="relative" style="height: 256px;">
                            <div id="chartLoading" class="absolute inset-0 flex items-center justify-center bg-gray-50 rounded-lg">
                                <div class="text-center">
                                    <div class="inline-block animate-spin rounded-full h-8 w-8 border-b-2 border-black mb-2"></div>
                                    <p class="text-xs text-gray-500">Loading chart data...</p>
                                </div>
                            </div>
                            <div id="chartError" class="hidden absolute inset-0 flex items-center justify-center bg-red-50 rounded-lg">
                                <div class="text-center p-4">
                                    <p class="text-sm text-red-600 font-medium">Unable to load chart data</p>
                                    <p class="text-xs text-red-500 mt-1">Please try refreshing the page</p>
                                </div>
                            </div>
                            <canvas id="stockChart" class="w-full h-full"></canvas>
                        </div>
                    </div>

                    <!-- Stock Details -->
                    <div class="bg-white rounded-xl p-6 shadow-sm border border-gray-100">
                        <div class="flex items-center justify-between mb-4">
                            <div>
                                <h3 class="text-lg font-light text-black mb-1">Stock Details</h3>
                                <p class="text-xs text-gray-500">Comprehensive stock information</p>
                            </div>
                        </div>
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div class="space-y-4">
                                <h4 class="text-sm font-medium text-black mb-3">Key Metrics</h4>
                                <div class="space-y-3">
                                    <div class="flex items-center justify-between">
                                        <span class="text-xs text-gray-600">Previous Close</span>
                                        <span class="text-xs font-medium text-black">$<?= number_format($prev_close, 2) ?></span>
                                    </div>
                                    <div class="flex items-center justify-between">
                                        <span class="text-xs text-gray-600">P/E Ratio</span>
                                        <span class="text-xs font-medium text-black">28.50</span>
                                    </div>
                                    <div class="flex items-center justify-between">
                                        <span class="text-xs text-gray-600">Dividend Yield</span>
                                        <span class="text-xs font-medium text-black">0.50%</span>
                                    </div>
                                    <div class="flex items-center justify-between">
                                        <span class="text-xs text-gray-600">52 Week High</span>
                                        <span class="text-xs font-medium text-black">$220.00</span>
                                    </div>
                                    <div class="flex items-center justify-between">
                                        <span class="text-xs text-gray-600">52 Week Low</span>
                                        <span class="text-xs font-medium text-black">$150.00</span>
                                    </div>
                                </div>
                            </div>
                            <div class="space-y-4">
                                <h4 class="text-sm font-medium text-black mb-3">Company Information</h4>
                                <div class="space-y-3">
                                    <div class="flex items-center justify-between">
                                        <span class="text-xs text-gray-600">Symbol</span>
                                        <span class="text-xs font-medium text-black"><?= htmlspecialchars($symbol) ?></span>
                                    </div>
                                    <div class="flex items-center justify-between">
                                        <span class="text-xs text-gray-600">Company</span>
                                        <span class="text-xs font-medium text-black"><?= htmlspecialchars($name) ?></span>
                                    </div>
                                    <div class="flex items-center justify-between">
                                        <span class="text-xs text-gray-600">Sector</span>
                                        <span class="text-xs font-medium text-black">Technology</span>
                                    </div>
                                    <div class="flex items-center justify-between">
                                        <span class="text-xs text-gray-600">Industry</span>
                                        <span class="text-xs font-medium text-black">Technology</span>
                                    </div>
                                    <div class="flex items-center justify-between">
                                        <span class="text-xs text-gray-600">Last Updated</span>
                                        <span class="text-xs font-medium text-black"><?= date('M j, Y g:i A') ?></span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                </div>

                <!-- Trading Actions + Recent Activity -->
                <div class="space-y-6">
                    <!-- Trading Actions -->
                    <div class="bg-white rounded-xl p-6 shadow-sm border border-gray-100">
                        <div class="flex items-center justify-between mb-4">
                            <div>
                                <h3 class="text-lg font-light text-black mb-1">Trading Actions</h3>
                                <p class="text-xs text-gray-500">Buy or sell this stock</p>
                            </div>
                        </div>
                        <div class="space-y-3">
                            <a href="stocks_buy.php?stock_id=<?= $stock_id ?>"
                               class="w-full flex items-center justify-center px-4 py-3 bg-green-600 text-white text-sm font-medium rounded-lg hover:bg-green-700 transition-colors duration-200">
                                <i data-lucide="plus" class="w-4 h-4 mr-2"></i>
                                Buy <?= htmlspecialchars($symbol) ?>
                            </a>
                        </div>
                    </div>

                    <!-- Recent Activity -->
                    <div class="bg-white rounded-xl p-6 shadow-sm border border-gray-100">
                        <div class="flex items-center justify-between mb-4">
                            <div>
                                <h3 class="text-lg font-light text-black mb-1">Recent Activity</h3>
                                <p class="text-xs text-gray-500">Latest transactions</p>
                            </div>
                        </div>
                        <div class="space-y-3">
                            <?php if (empty($transactions)): ?>
                                <p class="text-xs text-gray-500 text-center py-4">No recent activity</p>
                            <?php else: ?>
                                <?php foreach ($transactions as $t): ?>
                                    <?php
                                    $is_buy = $t['type'] === 'buy';
                                    $icon = $is_buy ? 'plus' : 'minus';
                                    $color = $is_buy ? 'text-green-500' : 'text-red-500';
                                    $amount = $t['total_amount'] + $t['fee'];
                                    $date = date('M j, Y', strtotime($t['created_at']));
                                    ?>
                                    <div class="flex items-center space-x-3 p-3 bg-gray-50 rounded-lg">
                                        <div class="w-6 h-6 flex items-center justify-center">
                                            <i data-lucide="<?= $icon ?>" class="w-3 h-3 <?= $color ?>"></i>
                                        </div>
                                        <div class="flex-1 min-w-0">
                                            <p class="text-xs font-medium text-black"><?= ucfirst($t['type']) ?></p>
                                            <p class="text-xs text-gray-500"><?= $date ?></p>
                                        </div>
                                        <div class="text-right">
                                            <p class="text-xs font-medium text-black">$<?= number_format($amount, 2) ?></p>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Chart.js + Logic -->
        <script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>
        <script>
            document.addEventListener('DOMContentLoaded', function () {
                if (typeof Chart === 'undefined') {
                    document.getElementById('chartError').classList.remove('hidden');
                    return;
                }

                const ctx = document.getElementById('stockChart');
                const stockChart = new Chart(ctx, {
                    type: 'line',
                    data: { labels: [], datasets: [{ label: 'Price', data: [], borderColor: '#000000', backgroundColor: 'rgba(0,0,0,0.1)', borderWidth: 2, fill: true, tension: 0.4, pointRadius: 0, pointHoverRadius: 6, pointHoverBackgroundColor: '#000000', pointHoverBorderColor: '#ffffff', pointHoverBorderWidth: 2 }] },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        interaction: { intersect: false, mode: 'index' },
                        plugins: {
                            legend: { display: false },
                            tooltip: {
                                backgroundColor: 'rgba(0,0,0,0.8)', titleColor: '#fff', bodyColor: '#fff', borderColor: '#000', borderWidth: 1, cornerRadius: 8, displayColors: false,
                                callbacks: { label: ctx => '$' + ctx.parsed.y.toFixed(2) }
                            }
                        },
                        scales: {
                            x: { display: false, grid: { display: false } },
                            y: {
                                display: true, position: 'right',
                                grid: { color: 'rgba(0,0,0,0.1)', drawBorder: false },
                                ticks: { color: '#6b7280', font: { size: 12 }, callback: v => '$' + v.toFixed(2) }
                            }
                        }
                    }
                });

                function loadChartData(period) {
                    document.getElementById('chartLoading').classList.remove('hidden');
                    document.getElementById('chartError').classList.add('hidden');

                    fetch(`/api/stocks/<?= $symbol ?>/chart-data?period=${period}`, {
                        headers: { 'Accept': 'application/json', 'X-Requested-With': 'XMLHttpRequest' },
                        credentials: 'same-origin'
                    })
                    .then(r => { if (!r.ok) throw new Error(r.status); return r.json(); })
                    .then(data => {
                        document.getElementById('chartLoading').classList.add('hidden');
                        if (data.success && data.chartData?.labels?.length > 0) {
                            stockChart.data.labels = data.chartData.labels;
                            stockChart.data.datasets[0].data = data.chartData.datasets[0].data;
                            if (data.chartData.datasets[0].borderColor) stockChart.data.datasets[0].borderColor = data.chartData.datasets[0].borderColor;
                            if (data.chartData.datasets[0].backgroundColor) stockChart.data.datasets[0].backgroundColor = data.chartData.datasets[0].backgroundColor;
                            stockChart.update('none');
                        } else {
                            throw new Error('No data');
                        }
                    })
                    .catch(() => {
                        document.getElementById('chartLoading').classList.add('hidden');
                        document.getElementById('chartError').classList.remove('hidden');
                    });
                }

                document.querySelectorAll('[data-period]').forEach(btn => {
                    btn.addEventListener('click', function () {
                        const period = this.dataset.period;
                        loadChartData(period);
                        document.querySelectorAll('[data-period]').forEach(b => {
                            b.classList.remove('bg-black', 'text-white');
                            b.classList.add('text-gray-600');
                        });
                        this.classList.remove('text-gray-600');
                        this.classList.add('bg-black', 'text-white');
                    });
                });

                loadChartData('1m'); // default
            });

            function editWatchlistAlert(symbol, currentPrice, currentType) {
                const modal = document.createElement('div');
                modal.className = 'fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50';
                modal.innerHTML = `
                    <div class="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
                        <div class="mt-3">
                            <h3 class="text-lg font-medium text-gray-900 mb-4">Edit Price Alert</h3>
                            <form action="https://tesla.cycloverseworld.info/trading/watchlist/update/1" method="POST">
                                <input type="hidden" name="_token" value="HkPgnpzcNQsvRAYTgBb4ntsiyoHWMRV3LF98KVWu">
                                <input type="hidden" name="_method" value="PATCH">
                                <div class="mb-4">
                                    <label for="alert_price" class="block text-sm font-medium text-gray-700 mb-2">Alert Price</label>
                                    <input type="number" id="alert_price" name="alert_price" value="${currentPrice || ''}" step="0.01" min="0.01"
                                           class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-black focus:border-transparent" placeholder="0.00">
                                </div>
                                <div class="mb-6">
                                    <label for="alert_type" class="block text-sm font-medium text-gray-700 mb-2">Alert Type</label>
                                    <select id="alert_type" name="alert_type" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-black focus:border-transparent">
                                        <option value="">No alert</option>
                                        <option value="above" ${currentType === 'above' ? 'selected' : ''}>Above this price</option>
                                        <option value="below" ${currentType === 'below' ? 'selected' : ''}>Below this price</option>
                                    </select>
                                </div>
                                <div class="flex justify-end space-x-3">
                                    <button type="button" onclick="this.closest('.fixed').remove()" class="px-4 py-2 text-gray-600 bg-gray-100 rounded-lg hover:bg-gray-200">Cancel</button>
                                    <button type="submit" class="px-4 py-2 bg-black text-white rounded-lg hover:bg-gray-800">Update Alert</button>
                                </div>
                            </form>
                        </div>
                    </div>`;
                document.body.appendChild(modal);
                modal.addEventListener('click', e => { if (e.target === modal) modal.remove(); });
            }
        </script>
    </main>
</div>

<?php include 'footer.php'; ?>