<?php
include 'header.php';

// -------------------------------------------------
// 3. FILTERS FROM GET
// -------------------------------------------------
$type_filter = $_GET['type'] ?? ''; // deposit or withdrawal
$status_filter = $_GET['status'] ?? '';
$date_from_filter = $_GET['date_from'] ?? '';
// -------------------------------------------------
// 4. BUILD DYNAMIC SQL QUERY - ONLY DEPOSITS & WITHDRAWALS
// -------------------------------------------------
$sql = "
    SELECT
        t.type,
        t.amount,
        t.fee,
        t.crypto_method,
        t.transaction_hash,
        t.address,
        t.status,
        t.created_at
    FROM transactions t
    WHERE t.user_id = ?
      AND t.type IN ('deposit', 'withdrawal')
";
$params = [$user_id];
$types = 'i';

if (!empty($type_filter)) {
    $sql .= " AND t.type = ?";
    $params[] = $type_filter;
    $types .= 's';
}
if (!empty($status_filter)) {
    $sql .= " AND t.status = ?";
    $params[] = $status_filter;
    $types .= 's';
}
if (!empty($date_from_filter)) {
    $sql .= " AND DATE(t.created_at) >= ?";
    $params[] = $date_from_filter;
    $types .= 's';
}

$sql .= " ORDER BY t.created_at DESC";

$stmt = mysqli_prepare($link, $sql);
if ($stmt === false) {
    die("Prepare failed: " . mysqli_error($link));
}
mysqli_stmt_bind_param($stmt, $types, ...$params);
mysqli_stmt_execute($stmt);
$result = mysqli_stmt_get_result($stmt);

// -------------------------------------------------
// 5. FETCH & CALCULATE STATS
// -------------------------------------------------
$transactions = [];
$total_transactions = 0;
$deposit_count = 0;
$withdrawal_count = 0;
$total_deposits = 0.0;
$total_withdrawals = 0.0;
$total_fees = 0.0;

while ($row = mysqli_fetch_assoc($result)) {
    $transactions[] = $row;
    $total_transactions++;

    if ($row['type'] === 'deposit') {
        $deposit_count++;
        $total_deposits += (float)$row['amount'];
    } elseif ($row['type'] === 'withdrawal') {
        $withdrawal_count++;
        $total_withdrawals += (float)$row['amount'];
    }
    $total_fees += (float)$row['fee'];
}

// -------------------------------------------------
// 6. FORMAT DATE/TIME HELPERS
// -------------------------------------------------
function formatDateTime($datetime) {
    $dt = new DateTime($datetime, new DateTimeZone('UTC'));
    $dt->setTimezone(new DateTimeZone('Africa/Lagos')); // WAT
    return [
        'date' => $dt->format('M j, Y'),
        'time' => $dt->format('g:i A')
    ];
}

mysqli_close($link);
?>
<!-- ============================================= -->
<!-- PAGE CONTENT -->
<!-- ============================================= -->
<div class="flex-1 flex flex-col overflow-hidden lg:ml-72">
    <!-- Top Bar -->
    <header class="bg-white border-b border-gray-200">
        <div class="flex items-center justify-between px-6 py-3">
            <div class="flex items-center">
                <button onclick="toggleSidebar()" class="lg:hidden text-gray-500 hover:text-gray-700 p-1.5 rounded-md hover:bg-gray-100 mr-3">
                    <i data-lucide="menu" class="w-5 h-5"></i>
                </button>
                <div class="font-light text-lg text-black">Deposit & Withdrawal History</div>
            </div>
        </div>
    </header>

    <!-- Page Content -->
    <main class="flex-1 overflow-y-auto p-6 bg-white pb-20 lg:pb-6">
        <div class="max-w-7xl mx-auto">
            <!-- Enhanced Header -->
            <div class="bg-gradient-to-br from-black via-gray-900 to-gray-800 rounded-2xl p-6 mb-6 text-white relative overflow-hidden">
                <div class="absolute inset-0 opacity-10">
                    <div class="absolute top-0 right-0 w-48 h-48 bg-white rounded-full -translate-y-24 translate-x-24"></div>
                    <div class="absolute bottom-0 left-0 w-24 h-24 bg-white rounded-full translate-y-12 -translate-x-12"></div>
                </div>
                <div class="relative z-10">
                    <div class="flex flex-col lg:flex-row lg:items-center lg:justify-between">
                        <div class="mb-4 lg:mb-0 lg:flex-1">
                            <h1 class="text-xl font-light mb-1">Deposit & Withdrawal History</h1>
                            <p class="text-gray-300 text-sm">Track all your funding and withdrawal activities</p>
                        </div>
                        <div class="bg-white bg-opacity-15 backdrop-blur-xl rounded-xl p-4 border border-white border-opacity-20 shadow-xl lg:w-80">
                            <div class="flex items-center justify-between mb-3">
                                <div>
                                    <p class="text-xs text-gray-300 mb-1">Total Transactions</p>
                                    <p class="text-lg font-light"><?= $total_transactions ?></p>
                                </div>
                                <div class="w-10 h-10 flex items-center justify-center">
                                    <i data-lucide="activity" class="w-5 h-5 text-white"></i>
                                </div>
                            </div>
                            <div class="flex space-x-4 text-xs">
                                <div class="flex-1 text-center">
                                    <p class="text-gray-300">Deposits</p>
                                    <p class="text-white font-medium"><?= $deposit_count ?></p>
                                </div>
                                <div class="flex-1 text-center">
                                    <p class="text-gray-300">Withdrawals</p>
                                    <p class="text-white font-medium"><?= $withdrawal_count ?></p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Filters -->
            <div class="bg-white rounded-xl p-4 shadow-sm border border-gray-100 mb-6">
                <form method="GET" class="grid grid-cols-1 md:grid-cols-4 gap-4">
                    <div>
                        <label for="type" class="block text-xs font-medium text-gray-700 mb-1">Type</label>
                        <select id="type" name="type" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-black focus:border-transparent">
                            <option value="">All Types</option>
                            <option value="deposit" <?= $type_filter === 'deposit' ? 'selected' : '' ?>>Deposit</option>
                            <option value="withdrawal" <?= $type_filter === 'withdrawal' ? 'selected' : '' ?>>Withdrawal</option>
                        </select>
                    </div>
                    <div>
                        <label for="status" class="block text-xs font-medium text-gray-700 mb-1">Status</label>
                        <select id="status" name="status" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-black focus:border-transparent">
                            <option value="">All Status</option>
                            <option value="confirmed" <?= $status_filter === 'confirmed' ? 'selected' : '' ?>>Completed</option>
                            <option value="pending" <?= $status_filter === 'pending' ? 'selected' : '' ?>>Pending</option>
                           
                        </select>
                    </div>
                    <div>
                        <label for="date_from" class="block text-xs font-medium text-gray-700 mb-1">From Date</label>
                        <input type="date" id="date_from" name="date_from" value="<?= htmlspecialchars($date_from_filter) ?>" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-black focus:border-transparent">
                    </div>
                    <div class="flex items-end space-x-2">
                        <button type="submit" class="px-4 py-2 bg-black text-white text-xs font-medium rounded-lg hover:bg-gray-800">
                            <i data-lucide="filter" class="w-3 h-3 mr-1"></i> Filter
                        </button>
                        <a href="?" class="px-4 py-2 border border-gray-300 text-black text-xs font-medium rounded-lg hover:bg-gray-50">
                            <i data-lucide="refresh-cw" class="w-3 h-3 mr-1"></i> Clear
                        </a>
                    </div>
                </form>
            </div>

            <!-- Transactions Table -->
            <div class="bg-white rounded-xl shadow-sm border border-gray-100 overflow-hidden">
                <div class="px-6 py-4 border-b border-gray-100">
                    <h3 class="text-lg font-light text-black mb-1">Deposit & Withdrawal Transactions</h3>
                    <p class="text-xs text-gray-500">All funding and cash-out activities</p>
                </div>
                <div class="overflow-x-auto">
                    <table class="w-full">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Date</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Type</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Crypto</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Amount</th>
           
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                            </tr>
                        </thead>
                        <tbody class="bg-white divide-y divide-gray-100">
                            <?php foreach ($transactions as $t): ?>
                                <?php $dt = formatDateTime($t['created_at']); ?>
                                <tr class="hover:bg-gray-50 transition-colors duration-200">
                                    <td class="px-6 py-4">
                                        <div class="text-sm text-black"><?= $dt['date'] ?></div>
                                        <div class="text-xs text-gray-500"><?= $dt['time'] ?></div>
                                    </td>
                                    <td class="px-6 py-4">
                                        <?php
                                            $type = $t['type'];
                                            $badge_class = $type === 'deposit' ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800';
                                            $label = ucfirst($type);
                                        ?>
                                        <span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium <?= $badge_class ?>">
                                            <?= $label ?>
                                        </span>
                                    </td>
                                    <td class="px-6 py-4">
                                        <span class="text-sm font-medium text-black"><?= strtoupper($t['crypto_method'] ?? 'N/A') ?></span>
                                    </td>
                                    <td class="px-6 py-4">
                                        <div class="text-sm text-black">$<?= number_format((float)$t['amount'], 2) ?></div>
                                    </td>
                                    
                                    <td class="px-6 py-4">
                                        <?php
                                            $status = $t['status'];
                                            $status_class = 'bg-green-100 text-green-800';
                                            $status_label = ucfirst($status);
                                            if ($status === 'pending') $status_class = 'bg-yellow-100 text-yellow-800';
                                            if ($status === 'cancelled') $status_class = 'bg-red-100 text-red-800';
                                        ?>
                                        <span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium <?= $status_class ?>">
                                            <?= $status_label ?>
                                        </span>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                            <?php if (empty($transactions)): ?>
                                <tr>
                                    <td colspan="7" class="px-6 py-12 text-center text-gray-500">
                                        No deposit or withdrawal transactions found.
                                    </td>
                                </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- Summary Stats -->
            <div class="grid grid-cols-1 md:grid-cols-4 gap-4 mt-6">
                <div class="bg-white rounded-xl p-4 shadow-sm border border-gray-100">
                    <div class="flex items-center justify-between mb-3">
                        <div>
                            <p class="text-xs text-gray-500 mb-1">Total Deposits</p>
                            <p class="text-lg font-light text-black">$<?= number_format($total_deposits, 2) ?></p>
                        </div>
                        <div class="w-8 h-8 flex items-center justify-center bg-green-100 rounded-lg">
                            <i data-lucide="arrow-down" class="w-4 h-4 text-green-600"></i>
                        </div>
                    </div>
                </div>
                <div class="bg-white rounded-xl p-4 shadow-sm border border-gray-100">
                    <div class="flex items-center justify-between mb-3">
                        <div>
                            <p class="text-xs text-gray-500 mb-1">Total Withdrawals</p>
                            <p class="text-lg font-light text-black">$<?= number_format($total_withdrawals, 2) ?></p>
                        </div>
                        <div class="w-8 h-8 flex items-center justify-center bg-red-100 rounded-lg">
                            <i data-lucide="arrow-up" class="w-4 h-4 text-red-600"></i>
                        </div>
                    </div>
                </div>
                <div class="bg-white rounded-xl p-4 shadow-sm border border-gray-100">
                    <div class="flex items-center justify-between mb-3">
                        <div>
                            <p class="text-xs text-gray-500 mb-1">Net Flow</p>
                            <p class="text-lg font-light text-black <?= ($total_deposits - $total_withdrawals >= 0 ? 'text-green-600' : 'text-red-600') ?>">
                                $<?= number_format($total_deposits - $total_withdrawals, 2) ?>
                            </p>
                        </div>
                        <div class="w-8 h-8 flex items-center justify-center">
                            <i data-lucide="trending-up" class="w-4 h-4 text-blue-600"></i>
                        </div>
                    </div>
                </div>
                
            </div>
        </div>
    </main>
</div>

<?php include 'footer.php'; ?>