<?php
include 'header.php';

// -------------------------------------------------------------------
// 1. Filters & Sorting
// -------------------------------------------------------------------
$search   = trim($_GET['search'] ?? '');
$sector   = $_GET['sector'] ?? '';
$industry = $_GET['industry'] ?? '';
$sort     = $_GET['sort'] ?? 'symbol';

$valid_sort = ['symbol', 'price', 'change', 'volume', 'market_cap'];
if (!in_array($sort, $valid_sort)) $sort = 'symbol';

$where = [];
$params = [];
$types = '';

if ($search !== '') {
    $where[] = "(symbol LIKE ? OR name LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $types .= 'ss';
}
if ($sector !== '') {
    $where[] = "sector = ?";
    $params[] = $sector;
    $types .= 's';
}
if ($industry !== '') {
    $where[] = "industry = ?";
    $params[] = $industry;
    $types .= 's';
}

$where_clause = $where ? 'WHERE ' . implode(' AND ', $where) : '';

// Sort map
$sort_map = [
    'symbol'     => 'symbol ASC',
    'price'      => 'price DESC',
    'change'     => '(price - previous_close)/previous_close DESC',
    'volume'     => 'volume DESC',
    'market_cap' => 'market_cap DESC'
];
$order_by = $sort_map[$sort] ?? 'symbol ASC';

// -------------------------------------------------------------------
// 2. Fetch stocks
// -------------------------------------------------------------------
$query = "SELECT id, symbol, name, sector, industry, logo_url, price, previous_close, volume, market_cap, updated_at
          FROM stocks $where_clause ORDER BY $order_by";

$stmt = mysqli_prepare($link, $query);
if ($params) {
    mysqli_stmt_bind_param($stmt, $types, ...$params);
}
mysqli_stmt_execute($stmt);
$result = mysqli_stmt_get_result($stmt);
$stocks = mysqli_fetch_all($result, MYSQLI_ASSOC);
mysqli_stmt_close($stmt);

// -------------------------------------------------------------------
// 3. Stats
// -------------------------------------------------------------------
$total_stocks = count($stocks);
$gainers = 0; $losers = 0;
$top_gainer_pct = -999; $top_loser_pct = 999;
$top_gainer_sym = ''; $top_loser_sym = '';
$most_active_vol = 0; $most_active_sym = '';

foreach ($stocks as $s) {
    $change_pct = (($s['price'] - $s['previous_close']) / $s['previous_close']) * 100;
    if ($change_pct > 0) $gainers++;
    if ($change_pct < 0) $losers++;

    if ($change_pct > $top_gainer_pct) {
        $top_gainer_pct = $change_pct;
        $top_gainer_sym = $s['symbol'];
    }
    if ($change_pct < $top_loser_pct) {
        $top_loser_pct = $change_pct;
        $top_loser_sym = $s['symbol'];
    }
    if ($s['volume'] > $most_active_vol) {
        $most_active_vol = $s['volume'];
        $most_active_sym = $s['symbol'];
    }
}

// Format numbers
function fmt($num, $dec = 2) { return number_format($num, $dec); }
function fmt_vol($v) { return $v >= 1_000_000 ? fmt($v/1_000_000, 1).'M' : fmt($v); }
function fmt_cap($c) { return '$'.fmt($c/1_000_000, 0).'M'; }

$last_updated = $stocks ? (new DateTime($stocks[0]['updated_at']))->format('H:i:s') : date('H:i:s');
?>

<div class="flex-1 flex flex-col overflow-hidden lg:ml-72">
    <header class="bg-white border-b border-gray-200">
        <div class="flex items-center justify-between px-6 py-3">
            <div class="flex items-center">
                <button onclick="toggleSidebar()" class="lg:hidden text-gray-500 hover:text-gray-700 p-1.5 rounded-md hover:bg-gray-100 mr-3">
                    <i data-lucide="menu" class="w-5 h-5"></i>
                </button>
                <div class="font-light text-lg text-black">Stock Marketplace</div>
            </div>
            
        </div>
    </header>

    <main class="flex-1 overflow-y-auto p-6 bg-white pb-20 lg:pb-6">
        <div class="max-w-7xl mx-auto">

            <!-- Header -->
            <div class="bg-gradient-to-br from-black via-gray-900 to-gray-800 rounded-2xl p-6 mb-6 text-white relative overflow-hidden">
                <div class="absolute inset-0 opacity-10">
                    <div class="absolute top-0 right-0 w-48 h-48 bg-white rounded-full -translate-y-24 translate-x-24"></div>
                    <div class="absolute bottom-0 left-0 w-24 h-24 bg-white rounded-full translate-y-12 -translate-x-12"></div>
                </div>
                <div class="relative z-10">
                    <div class="flex flex-col lg:flex-row lg:items-center lg:justify-between">
                        <div class="mb-4 lg:mb-0 lg:flex-1">
                            <div class="flex items-center mb-1">
                                <h1 class="text-xl font-light">Stock Marketplace</h1>
                                <div class="ml-3 real-time-indicator">
                                    <span class="inline-flex items-center px-2 py-0.5 rounded-full text-xs font-medium bg-green-900 text-green-300">
                                        <i data-lucide="radio" class="w-3 h-3 mr-1"></i> Live
                                    </span>
                                </div>
                            </div>
                            <p class="text-gray-300 text-sm">Discover and trade stocks from leading companies</p>
                            <p class="text-xs text-gray-400 mt-1">Last updated: <span data-last-updated><?php echo $last_updated; ?></span></p>
                        </div>
                        <div class="bg-white bg-opacity-15 backdrop-blur-xl rounded-xl p-4 border border-white border-opacity-20 shadow-xl lg:w-80">
                            <div class="flex items-center justify-between mb-3">
                                <div>
                                    <p class="text-xs text-gray-300 mb-1">Active Stocks</p>
                                    <p class="text-lg font-light"><?php echo $total_stocks; ?></p>
                                </div>
                                <div class="w-10 h-10 flex items-center justify-center">
                                    <i data-lucide="bar-chart-3" class="w-5 h-5 text-white"></i>
                                </div>
                            </div>
                            <div class="flex space-x-4 text-xs">
                                <div class="flex-1 text-center">
                                    <p class="text-gray-300">Gainers</p>
                                    <p class="text-white font-medium"><?php echo $gainers; ?></p>
                                </div>
                                <div class="flex-1 text-center">
                                    <p class="text-gray-300">Losers</p>
                                    <p class="text-white font-medium"><?php echo $losers; ?></p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Market Overview -->
            <div class="grid grid-cols-1 md:grid-cols-3 gap-4 mb-6">
                <a href="?sort=change" class="block bg-white rounded-xl p-4 shadow-sm border border-gray-100 hover:shadow-md hover:border-green-200 transition-all duration-200 group">
                    <div class="flex items-center justify-between mb-3">
                        <div>
                            <p class="text-xs text-gray-500 mb-1">Top Gainer</p>
                            <p class="text-lg font-light text-green-600">+<?php echo number_format($top_gainer_pct, 2); ?>%</p>
                            <p class="text-xs text-gray-400 mt-1"><?php echo $top_gainer_sym; ?></p>
                        </div>
                        <i data-lucide="trending-up" class="w-4 h-4 text-green-600 group-hover:scale-110 transition-transform"></i>
                    </div>
                </a>

                <a href="?sort=change" class="block bg-white rounded-xl p-4 shadow-sm border border-gray-100 hover:shadow-md hover:border-red-200 transition-all duration-200 group">
                    <div class="flex items-center justify-between mb-3">
                        <div>
                            <p class="text-xs text-gray-500 mb-1">Top Loser</p>
                            <p class="text-lg font-light text-red-600"><?php echo number_format($top_loser_pct, 2); ?>%</p>
                            <p class="text-xs text-gray-400 mt-1"><?php echo $top_loser_sym; ?></p>
                        </div>
                        <i data-lucide="trending-down" class="w-4 h-4 text-red-600 group-hover:scale-110 transition-transform"></i>
                    </div>
                </a>

                <a href="?sort=volume" class="block bg-white rounded-xl p-4 shadow-sm border border-gray-100 hover:shadow-md hover:border-blue-200 transition-all duration-200 group">
                    <div class="flex items-center justify-between mb-3">
                        <div>
                            <p class="text-xs text-gray-500 mb-1">Most Active</p>
                            <p class="text-lg font-light text-black"><?php echo fmt_vol($most_active_vol); ?></p>
                            <p class="text-xs text-gray-400 mt-1"><?php echo $most_active_sym; ?></p>
                        </div>
                        <i data-lucide="activity" class="w-4 h-4 text-blue-600 group-hover:scale-110 transition-transform"></i>
                    </div>
                </a>
            </div>

            <!-- Search & Filters -->
            <div class="bg-white rounded-xl p-4 shadow-sm border border-gray-100 mb-6">
                <form method="GET" class="grid grid-cols-1 md:grid-cols-5 gap-4">
                    <div>
                        <label for="search" class="block text-xs font-medium text-gray-700 mb-1">Search</label>
                        <input type="text" id="search" name="search" value="<?php echo htmlspecialchars($search); ?>" placeholder="Symbol or company name" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-black focus:border-transparent">
                    </div>
                    <div>
                        <label for="sector" class="block text-xs font-medium text-gray-700 mb-1">Sector</label>
                        <select id="sector" name="sector" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-black focus:border-transparent">
                            <option value="">All Sectors</option>
                            <?php
                            $sectors = array_unique(array_column($stocks, 'sector'));
                            foreach ($sectors as $s) {
                                $sel = $sector === $s ? 'selected' : '';
                                echo "<option value=\"$s\" $sel>$s</option>";
                            }
                            ?>
                        </select>
                    </div>
                    <div>
                        <label for="industry" class="block text-xs font-medium text-gray-700 mb-1">Industry</label>
                        <select id="industry" name="industry" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-black focus:border-transparent">
                            <option value="">All Industries</option>
                            <?php
                            $industries = array_unique(array_column($stocks, 'industry'));
                            foreach ($industries as $i) {
                                $sel = $industry === $i ? 'selected' : '';
                                echo "<option value=\"$i\" $sel>$i</option>";
                            }
                            ?>
                        </select>
                    </div>
                    <div>
                        <label for="sort" class="block text-xs font-medium text-gray-700 mb-1">Sort By</label>
                        <select id="sort" name="sort" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-black focus:border-transparent">
                            <option value="symbol" <?php echo $sort==='symbol'?'selected':'';?>>Symbol</option>
                            <option value="price" <?php echo $sort==='price'?'selected':'';?>>Price</option>
                            <option value="change" <?php echo $sort==='change'?'selected':'';?>>Change</option>
                            <option value="volume" <?php echo $sort==='volume'?'selected':'';?>>Volume</option>
                            <option value="market_cap" <?php echo $sort==='market_cap'?'selected':'';?>>Market Cap</option>
                        </select>
                    </div>
                    <div class="flex items-end space-x-2">
                        <button type="submit" class="px-4 py-2 bg-black text-white text-xs font-medium rounded-lg hover:bg-gray-800">
                            <i data-lucide="search" class="w-3 h-3 mr-1"></i> Search
                        </button>
                        <a href="?" class="px-4 py-2 border border-gray-300 text-black text-xs font-medium rounded-lg hover:bg-gray-50">
                            <i data-lucide="refresh-cw" class="w-3 h-3 mr-1"></i> Clear
                        </a>
                    </div>
                </form>
            </div>

            <!-- Featured Stocks (Top 3 by change) -->
            <div class="bg-white rounded-xl p-6 shadow-sm border border-gray-100 mb-6">
                <div class="flex items-center justify-between mb-4">
                    <div>
                        <h3 class="text-lg font-light text-black mb-1">Featured Stocks</h3>
                        <p class="text-xs text-gray-500">Handpicked stocks for your portfolio</p>
                    </div>
                </div>
                <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                    <?php
                    usort($stocks, fn($a,$b) => ($b['price']-$b['previous_close']) <=> ($a['price']-$a['previous_close']));
                    foreach (array_slice($stocks, 0, 3) as $i => $s):
                        $change = $s['price'] - $s['previous_close'];
                        $change_pct = ($change / $s['previous_close']) * 100;
                        $is_up = $change >= 0;
                    ?>
                    <div class="bg-gray-50 rounded-lg p-4 hover:bg-gray-100 transition-all duration-300">
                        <div class="flex items-center justify-between mb-3">
                            <div>
                                <h4 class="text-sm font-medium text-black"><?php echo $s['symbol']; ?></h4>
                                <p class="text-xs text-gray-500"><?php echo $s['name']; ?></p>
                            </div>
                            <div class="text-right">
                                <p class="text-sm font-medium text-black">$<?php echo fmt($s['price'], 2); ?></p>
                                <p class="text-xs <?php echo $is_up?'text-green-600':'text-red-600'; ?>">
                                    <?php echo $is_up?'+':''; ?>$<?php echo fmt(abs($change), 2); ?> (<?php echo $is_up?'+':''; ?><?php echo fmt(abs($change_pct), 2); ?>%)
                                </p>
                            </div>
                        </div>
                        <div class="flex items-center justify-between">
                            <span class="text-xs text-gray-500"><?php echo $s['sector']; ?></span>
                            <a href="view_stocks.php?id=<?php echo $s['id']; ?>" class="text-xs text-blue-600 hover:text-blue-800">View Details</a>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>

            <!-- All Stocks Table -->
            <div class="bg-white rounded-xl shadow-sm border border-gray-100 overflow-hidden">
                <div class="px-6 py-4 border-b border-gray-100">
                    <div class="flex items-center justify-between">
                        <div>
                            <h3 class="text-lg font-light text-black mb-1">All Stocks</h3>
                            <p class="text-xs text-gray-500"><?php echo $total_stocks; ?> stocks available</p>
                        </div>
                    </div>
                </div>
                <div class="overflow-x-auto">
                    <table class="w-full">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Stock</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Price</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Change</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Volume</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Market Cap</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                            </tr>
                        </thead>
                        <tbody class="bg-white divide-y divide-gray-100">
                            <?php foreach ($stocks as $s):
                                $change = $s['price'] - $s['previous_close'];
                                $change_pct = ($change / $s['previous_close']) * 100;
                                $is_up = $change >= 0;
                            ?>
                            <tr class="hover:bg-gray-50 transition-colors duration-200">
                                <td class="px-6 py-4">
                                    <div class="flex items-center">
                                        <div class="flex-shrink-0 w-8 h-8 rounded-lg overflow-hidden">
                                            <img src="<?php echo $s['logo_url']; ?>" alt="<?php echo $s['name']; ?>" class="w-8 h-8 object-cover">
                                        </div>
                                        <div class="ml-3">
                                            <div class="text-sm font-medium text-black"><?php echo $s['symbol']; ?></div>
                                            <div class="text-xs text-gray-500"><?php echo $s['name']; ?></div>
                                        </div>
                                    </div>
                                </td>
                                <td class="px-6 py-4">
                                    <div class="text-sm text-black">$<?php echo fmt($s['price'], 2); ?></div>
                                </td>
                                <td class="px-6 py-4">
                                    <div class="text-sm <?php echo $is_up?'text-green-600':'text-red-600'; ?>">
                                        <?php echo $is_up?'+':''; ?>$<?php echo fmt(abs($change), 2); ?>
                                    </div>
                                    <div class="text-xs <?php echo $is_up?'text-green-600':'text-red-600'; ?>">
                                        <?php echo $is_up?'+':''; ?><?php echo fmt(abs($change_pct), 2); ?>%
                                    </div>
                                </td>
                                <td class="px-6 py-4">
                                    <div class="text-sm text-black"><?php echo fmt_vol($s['volume']); ?></div>
                                </td>
                                <td class="px-6 py-4">
                                    <div class="text-sm text-black"><?php echo fmt_cap($s['market_cap']); ?></div>
                                </td>
                                <td class="px-6 py-4">
                                    <div class="flex space-x-2">
                                        <a href="view_stocks.php?id=<?php echo $s['id']; ?>" class="w-8 h-8 flex items-center justify-center text-blue-600 hover:text-blue-800 hover:bg-blue-50 rounded-lg" title="View Details">
                                            <i data-lucide="eye" class="w-4 h-4"></i>
                                        </a>
                                        <a href="stocks_buy.php?stock_id=<?php echo $s['id']; ?>" class="w-8 h-8 flex items-center justify-center text-green-600 hover:text-green-800 hover:bg-green-50 rounded-lg" title="Buy Stock">
                                            <i data-lucide="plus" class="w-4 h-4"></i>
                                        </a>
                                       
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </main>
</div>

<?php include 'footer.php'; ?>