<?php
include 'header.php';



// -------------------------------------------------
// 3. FETCH CURRENT PRICES FROM `stocks`
// -------------------------------------------------
$price_sql = "SELECT id, symbol, price FROM stocks";
$price_res = mysqli_query($link, $price_sql);
$stock_prices = [];
while ($row = mysqli_fetch_assoc($price_res)) {
    $stock_prices[$row['id']] = [
        'symbol' => $row['symbol'],
        'price'  => (float)$row['price']
    ];
}

// -------------------------------------------------
// 4. FETCH USER'S STOCK TRANSACTIONS
// -------------------------------------------------
$trans_sql = "
    SELECT 
        st.stock_id,
        s.symbol,
        s.name AS company_name,
        s.logo_url,
        st.type,
        st.shares,
        st.price_per_share,
        st.total_amount,
        st.fee,
        st.created_at,
        st.status
    FROM stock_transactions st
    JOIN stocks s ON st.stock_id = s.id
    WHERE st.user_id = ? 
    ORDER BY st.created_at DESC
";
$stmt = mysqli_prepare($link, $trans_sql);
mysqli_stmt_bind_param($stmt, "i", $user_id);
mysqli_stmt_execute($stmt);
$trans_res = mysqli_stmt_get_result($stmt);

// -------------------------------------------------
// 5. BUILD HOLDINGS + RECENT TRANSACTIONS
// -------------------------------------------------
$holdings = [];        // per stock: shares, invested, avg_cost
$recent_transactions = []; // last 5

while ($row = mysqli_fetch_assoc($trans_res)) {
    $sid = $row['stock_id'];
    $shares = (float)$row['shares'];
    $cost = (float)$row['total_amount'];

    // Initialize holding
    if (!isset($holdings[$sid])) {
        $holdings[$sid] = [
            'symbol' => $row['symbol'],
            'name'   => $row['company_name'],
            'logo'   => $row['logo_url'],
            'shares' => 0.0,
            'invested' => 0.0,
        ];
    }

    // Adjust shares and cost
    if ($row['type'] === 'sell') {
        $shares = -$shares;
        $cost = -$cost;
    }

    $holdings[$sid]['shares'] += $shares;
    $holdings[$sid]['invested'] += $cost;

    // Store recent (limit to 5)
    if (count($recent_transactions) < 5) {
        $recent_transactions[] = $row;
    }
}

// -------------------------------------------------
// 6. CALCULATE PORTFOLIO TOTALS
// -------------------------------------------------
$total_value = 0;
$total_invested = 0;
$total_gain = 0;
$num_holdings = 0;

foreach ($holdings as $sid => $h) {
    if ($h['shares'] <= 0) continue;
    $num_holdings++;

    $current_price = $stock_prices[$sid]['price'] ?? 0;
    $value = $h['shares'] * $current_price;
    $gain = $value - $h['invested'];

    $h['current_price'] = $current_price;
    $h['value'] = $value;
    $h['gain'] = $gain;
    $h['pct'] = $h['invested'] > 0 ? ($gain / $h['invested']) * 100 : 0;
    $h['avg_price'] = $h['shares'] > 0 ? $h['invested'] / $h['shares'] : 0;

    $holdings[$sid] = $h;

    $total_value += $value;
    $total_invested += $h['invested'];
    $total_gain += $gain;
}
$total_return_pct = $total_invested > 0 ? ($total_gain / $total_invested) * 100 : 0;

mysqli_close($link);
?>

<!-- ============================================= -->
<!--               PAGE CONTENT                    -->
<!-- ============================================= -->
<div class="flex-1 flex flex-col overflow-hidden lg:ml-72">
    <!-- Top Bar -->
    <header class="bg-white border-b border-gray-200">
        <div class="flex items-center justify-between px-6 py-3">
            <div class="flex items-center">
                <button onclick="toggleSidebar()" class="lg:hidden text-gray-500 hover:text-gray-700 p-1.5 rounded-md hover:bg-gray-100 mr-3">
                    <i data-lucide="menu" class="w-5 h-5"></i>
                </button>
                <div class="font-light text-lg text-black">Stock Portfolio</div>
            </div>
            
        </div>
    </header>

    <!-- Page Content -->
    <main class="flex-1 overflow-y-auto p-6 bg-white pb-20 lg:pb-6">
        <div class="max-w-7xl mx-auto">

            <!-- Portfolio Summary Cards -->
            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
                <div class="bg-white rounded-xl p-6 shadow-sm border border-gray-100">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm font-medium text-gray-600 mb-1">Total Value</p>
                            <p class="text-2xl font-bold text-gray-900">$<?= number_format($total_value, 2) ?></p>
                        </div>
                        <div class="w-12 h-12 bg-gradient-to-br from-blue-500 to-blue-600 rounded-lg flex items-center justify-center">
                            <i data-lucide="dollar-sign" class="w-6 h-6 text-white"></i>
                        </div>
                    </div>
                </div>

                <div class="bg-white rounded-xl p-6 shadow-sm border border-gray-100">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm font-medium text-gray-600 mb-1">Total Invested</p>
                            <p class="text-2xl font-bold text-gray-900">$<?= number_format($total_invested, 2) ?></p>
                        </div>
                        <div class="w-12 h-12 bg-gradient-to-br from-green-500 to-green-600 rounded-lg flex items-center justify-center">
                            <i data-lucide="trending-up" class="w-6 h-6 text-white"></i>
                        </div>
                    </div>
                </div>

                <div class="bg-white rounded-xl p-6 shadow-sm border border-gray-100">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm font-medium text-gray-600 mb-1">Total Gain/Loss</p>
                            <p class="text-2xl font-bold <?= $total_gain >= 0 ? 'text-green-600' : 'text-red-600' ?>">
                                <?= $total_gain >= 0 ? '+' : '' ?>$<?= number_format($total_gain, 2) ?>
                            </p>
                            <p class="text-sm <?= $total_return_pct >= 0 ? 'text-green-600' : 'text-red-600' ?>">
                                <?= $total_return_pct >= 0 ? '+' : '' ?><?= number_format($total_return_pct, 2) ?>%
                            </p>
                        </div>
                        <div class="w-12 h-12 bg-gradient-to-br from-green-500 to-green-600 rounded-lg flex items-center justify-center">
                            <i data-lucide="trending-up" class="w-6 h-6 text-white"></i>
                        </div>
                    </div>
                </div>

                <div class="bg-white rounded-xl p-6 shadow-sm border border-gray-100">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm font-medium text-gray-600 mb-1">Holdings</p>
                            <p class="text-2xl font-bold text-gray-900"><?= $num_holdings ?></p>
                        </div>
                        <div class="w-12 h-12 bg-gradient-to-br from-purple-500 to-purple-600 rounded-lg flex items-center justify-center">
                            <i data-lucide="pie-chart" class="w-6 h-6 text-white"></i>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Holdings Table -->
            <div class="bg-white rounded-xl shadow-sm border border-gray-100 mb-8">
                <div class="p-6 border-b border-gray-100">
                    <h2 class="text-lg font-semibold text-gray-900">Your Holdings</h2>
                    <p class="text-sm text-gray-600 mt-1">Track your stock investments and performance</p>
                </div>

                <div class="overflow-x-auto">
                    <table class="w-full">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Stock</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Shares</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Avg Price</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Current Price</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Total Value</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Gain/Loss</th>
                                
                            </tr>
                        </thead>
                        <tbody class="bg-white divide-y divide-gray-200">
                            <?php foreach ($holdings as $sid => $h): ?>
                                <?php if ($h['shares'] <= 0) continue; ?>
                                <?php
                                    $price_change = $h['current_price'] - ($h['avg_price'] > 0 ? $h['avg_price'] : $h['current_price']);
                                    $price_pct = $h['avg_price'] > 0 ? ($price_change / $h['avg_price']) * 100 : 0;
                                ?>
                                <tr class="hover:bg-gray-50">
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <div class="flex items-center">
                                            <img src="<?= htmlspecialchars($h['logo']) ?>" alt="<?= htmlspecialchars($h['symbol']) ?>" class="w-8 h-8 rounded mr-3">
                                            <div>
                                                <div class="text-sm font-medium text-gray-900"><?= htmlspecialchars($h['symbol']) ?></div>
                                                <div class="text-sm text-gray-500"><?= htmlspecialchars($h['name']) ?></div>
                                            </div>
                                        </div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900"><?= number_format($h['shares'], 4) ?></td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">$<?= number_format($h['avg_price'], 2) ?></td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <div class="text-sm font-medium text-gray-900">$<?= number_format($h['current_price'], 2) ?></div>
                                        <div class="text-xs <?= $price_pct >= 0 ? 'text-green-600' : 'text-red-600' ?>">
                                            <?= $price_pct >= 0 ? '+' : '' ?><?= number_format($price_pct, 2) ?>%
                                        </div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">$<?= number_format($h['value'], 2) ?></td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <div class="text-sm font-medium <?= $h['gain'] >= 0 ? 'text-green-600' : 'text-red-600' ?>">
                                            <?= $h['gain'] >= 0 ? '+' : '' ?>$<?= number_format($h['gain'], 2) ?>
                                        </div>
                                        <div class="text-xs <?= $h['pct'] >= 0 ? 'text-green-600' : 'text-red-600' ?>">
                                            <?= $h['pct'] >= 0 ? '+' : '' ?><?= number_format($h['pct'], 2) ?>%
                                        </div>
                                    </td>
                                    
                                </tr>
                            <?php endforeach; ?>
                            <?php if (empty($holdings) || $num_holdings == 0): ?>
                                <tr>
                                    <td colspan="7" class="px-6 py-8 text-center text-gray-500">No stock holdings yet.</td>
                                </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- Recent Transactions -->
            <div class="bg-white rounded-xl shadow-sm border border-gray-100">
                <div class="p-6 border-b border-gray-100">
                    <h2 class="text-lg font-semibold text-gray-900">Recent Transactions</h2>
                    <p class="text-sm text-gray-600 mt-1">Your latest stock trading activity</p>
                </div>

                <div class="overflow-x-auto">
                    <table class="w-full">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Date</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Stock</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Type</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Quantity</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Price</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Total</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                            </tr>
                        </thead>
                        <tbody class="bg-white divide-y divide-gray-200">
                            <?php foreach ($recent_transactions as $t): ?>
                                <tr class="hover:bg-gray-50">
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                        <?= date('M j, Y', strtotime($t['created_at'])) ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <div class="flex items-center">
                                            <img src="<?= htmlspecialchars($t['logo_url']) ?>" alt="<?= htmlspecialchars($t['symbol']) ?>" class="w-6 h-6 rounded mr-2">
                                            <span class="text-sm font-medium text-gray-900"><?= htmlspecialchars($t['symbol']) ?></span>
                                        </div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium <?= $t['type'] === 'buy' ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800' ?>">
                                            <?= ucfirst($t['type']) ?>
                                        </span>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900"><?= number_format($t['shares'], 4) ?></td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">$<?= number_format($t['price_per_share'], 2) ?></td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">$<?= number_format($t['total_amount'], 2) ?></td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
                                            <?= ucfirst($t['status']) ?>
                                        </span>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                            <?php if (empty($recent_transactions)): ?>
                                <tr>
                                    <td colspan="7" class="px-6 py-8 text-center text-gray-500">No recent transactions.</td>
                                </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>

    
            </div>

        </div>
    </main>
</div>

<?php include 'footer.php'; ?>