<?php
include 'header.php';

/* -------------------------------------------------
   1. Get the plan
   ------------------------------------------------- */
$plan_id = (int)($_GET['id'] ?? 0);
if ($plan_id <= 0) die('Invalid plan.');

$stmt = mysqli_prepare($link, "
    SELECT name, category, risk_level, nav, min_investment
    FROM plans WHERE id = ?
");
mysqli_stmt_bind_param($stmt, "i", $plan_id);
mysqli_stmt_execute($stmt);
mysqli_stmt_bind_result($stmt, $plan_name, $plan_category, $plan_risk, $plan_nav, $plan_min);
if (!mysqli_stmt_fetch($stmt)) die('Plan not found.');
mysqli_stmt_close($stmt);

/* -------------------------------------------------
   2. Get user holdings for this plan
   ------------------------------------------------- */
$holding_units = 0.0;
$holding_value = 0.0;

$stmt = mysqli_prepare($link, "
    SELECT SUM(units) AS total_units
    FROM transactions
    WHERE user_id = ? AND plan_id = ? AND type = 'investment' AND status = 'confirmed'
");
mysqli_stmt_bind_param($stmt, "ii", $user_id, $plan_id);
mysqli_stmt_execute($stmt);
mysqli_stmt_bind_result($stmt, $total_units);
if (mysqli_stmt_fetch($stmt)) {
    $holding_units = $total_units ?? 0.0;
}
mysqli_stmt_close($stmt);

$holding_value = $holding_units * $plan_nav;

/* -------------------------------------------------
   3. Get recent activity (last 5 buys/sells)
   ------------------------------------------------- */
$activity = [];
$stmt = mysqli_prepare($link, "
    SELECT type, amount, total_cost, units, created_at
    FROM transactions
    WHERE user_id = ? AND plan_id = ?
      AND status = 'confirmed'
    ORDER BY created_at DESC
    LIMIT 5
");
mysqli_stmt_bind_param($stmt, "ii", $user_id, $plan_id);
mysqli_stmt_execute($stmt);
$result = mysqli_stmt_get_result($stmt);
while ($row = mysqli_fetch_assoc($result)) {
    $activity[] = $row;
}
mysqli_stmt_close($stmt);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title><?php echo htmlspecialchars($plan_name); ?> – View</title>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <!-- Lucide icons (already loaded in header.php) -->
</head>
<body class="bg-gray-50">

<div class="flex-1 flex flex-col overflow-hidden lg:ml-72">
    <!-- Top Bar -->
    <header class="bg-white border-b border-gray-200">
        <div class="flex items-center justify-between px-6 py-3">
            <div class="flex items-center">
                <button onclick="toggleSidebar()" class="lg:hidden text-gray-500 hover:text-gray-700 p-1.5 rounded-md hover:bg-gray-100 mr-3">
                    <i data-lucide="menu" class="w-5 h-5"></i>
                </button>
                <div class="font-light text-lg text-black">
                    <?php echo htmlspecialchars($plan_name); ?>
                </div>
            </div>

            <!-- Notifications (kept exactly as you had) -->
            
        </div>
    </header>

    <!-- Page Content -->
    <main class="flex-1 overflow-y-auto p-6 bg-white pb-20 lg:pb-6">
        <div class="max-w-7xl mx-auto">

            <!-- Header Card -->
            <div class="bg-gradient-to-br from-black via-gray-900 to-gray-800 rounded-2xl p-6 mb-6 text-white relative overflow-hidden">
                <div class="absolute inset-0 opacity-10">
                    <div class="absolute top-0 right-0 w-48 h-48 bg-white rounded-full -translate-y-24 translate-x-24"></div>
                    <div class="absolute bottom-0 left-0 w-24 h-24 bg-white rounded-full translate-y-12 -translate-x-12"></div>
                </div>
                <div class="relative z-10">
                    <div class="flex flex-col lg:flex-row lg:items-center lg:justify-between">
                        <div class="mb-4 lg:mb-0 lg:flex-1">
                            <h1 class="text-xl font-light mb-1"><?php echo htmlspecialchars($plan_name); ?></h1>
                            <p class="text-gray-300 text-sm">
                                <?php echo htmlspecialchars($plan_category); ?> • <?php echo ucfirst($plan_risk); ?> Risk
                            </p>
                        </div>

                        <div class="bg-white bg-opacity-15 backdrop-blur-xl rounded-xl p-4 border border-white border-opacity-20 shadow-xl lg:w-80">
                            <div class="flex items-center justify-between mb-3">
                                <div>
                                    <p class="text-xs text-gray-300 mb-1">Current NAV</p>
                                    <p class="text-lg font-light">$<?php echo number_format($plan_nav, 4); ?></p>
                                </div>
                                <div class="w-10 h-10 flex items-center justify-center">
                                    <i data-lucide="trending-up" class="w-5 h-5 text-white"></i>
                                </div>
                            </div>
                            <div class="flex space-x-4 text-xs">
                                <div class="flex-1 text-center">
                                    <p class="text-gray-300">Change</p>
                                    <p class="text-white font-medium text-green-400">+9.23%</p>
                                </div>
                                <div class="flex-1 text-center">
                                    <p class="text-gray-300">Min Investment</p>
                                    <p class="text-white font-medium">$<?php echo number_format($plan_min, 2); ?></p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">

                <!-- LEFT COLUMN -->
                <div class="lg:col-span-2 space-y-6">

                    <!-- Plan Overview -->
                    <div class="bg-white rounded-xl p-6 shadow-sm border border-gray-100">
                        <div class="flex items-center justify-between mb-4">
                            <h2 class="text-lg font-light text-black">Plan Overview</h2>
                            <span class="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium <?php
                                echo $plan_risk === 'high' ? 'bg-red-100 text-red-800' :
                                    ($plan_risk === 'medium' ? 'bg-yellow-100 text-yellow-800' : 'bg-green-100 text-green-800');
                            ?>">
                                <?php echo ucfirst($plan_risk); ?> Risk
                            </span>
                        </div>

                        <div class="grid grid-cols-2 md:grid-cols-4 gap-4 mb-6">
                            <div class="text-center">
                                <p class="text-xs text-gray-500 mb-1">NAV</p>
                                <p class="text-lg font-medium text-black">$<?php echo number_format($plan_nav, 4); ?></p>
                            </div>
                            <div class="text-center">
                                <p class="text-xs text-gray-500 mb-1">1Y Return</p>
                                <p class="text-lg font-medium text-green-600">+9.23%</p>
                            </div>
                            <div class="text-center">
                                <p class="text-xs text-gray-500 mb-1">Min Investment</p>
                                <p class="text-lg font-medium text-black">$<?php echo number_format($plan_min, 2); ?></p>
                            </div>
                            <div class="text-center">
                                <p class="text-xs text-gray-500 mb-1">Management Fee</p>
                                <p class="text-lg font-medium text-black">0.85%</p>
                            </div>
                        </div>

                        <div class="prose prose-sm max-w-none">
                            <p class="text-gray-600">
                                High-growth fund focused on <?php echo htmlspecialchars($plan_name); ?> and related sectors.
                            </p>
                        </div>
                    </div>

                    <!-- Performance Chart -->
                    <div class="bg-white rounded-xl p-6 shadow-sm border border-gray-100">
                        <div class="flex items-center justify-between mb-4">
                            <h2 class="text-lg font-light text-black">Performance</h2>
                            <div class="flex space-x-2">
                                <button data-period="1y" class="period-btn px-3 py-1 text-xs font-medium rounded bg-black text-white">1Y</button>
                                <button data-period="3y" class="period-btn px-3 py-1 text-xs font-medium rounded bg-gray-100 text-gray-600 hover:bg-gray-200">3Y</button>
                                <button data-period="5y" class="period-btn px-3 py-1 text-xs font-medium rounded bg-gray-100 text-gray-600 hover:bg-gray-200">5Y</button>
                            </div>
                        </div>
                        <div class="relative">
                            <canvas id="investmentChart" class="w-full h-64"></canvas>
                        </div>
                    </div>

                    <!-- Holdings Breakdown (static – replace with real data if you have) -->
                    <div class="bg-white rounded-xl p-6 shadow-sm border border-gray-100">
                        <h2 class="text-lg font-light text-black mb-4">Holdings Breakdown</h2>
                        <div class="space-y-3">
                            <div class="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                                <div class="flex items-center">
                                    <div class="w-8 h-8 flex items-center justify-center mr-3">
                                        <i data-lucide="building" class="w-4 h-4 text-blue-500"></i>
                                    </div>
                                    <div>
                                        <p class="text-sm font-medium text-black">Technology</p>
                                        <p class="text-xs text-gray-500">Tesla, Apple, Microsoft</p>
                                    </div>
                                </div>
                                <div class="text-right">
                                    <p class="text-sm font-medium text-black">45.2%</p>
                                    <p class="text-xs text-gray-500">$2.3M</p>
                                </div>
                            </div>
                            <!-- repeat for other rows -->
                        </div>
                    </div>

                </div>

                <!-- RIGHT SIDEBAR -->
                <div class="space-y-6">

                    <!-- Invest Now -->
                    <div class="bg-white rounded-xl p-6 shadow-sm border border-gray-100">
                        <h3 class="text-lg font-light text-black mb-4">Invest Now</h3>

                        <div class="mb-4 p-3 bg-blue-50 rounded-lg border border-blue-200">
                            <div class="flex items-center justify-between mb-2">
                                <span class="text-xs font-medium text-blue-800">Your Holdings</span>
                                <div class="w-6 h-6 flex items-center justify-center">
                                    <i data-lucide="pie-chart" class="w-4 h-4 text-blue-600"></i>
                                </div>
                            </div>
                            <div class="space-y-1">
                                <div class="flex justify-between">
                                    <span class="text-xs text-blue-700">Units:</span>
                                    <span class="text-xs font-medium text-blue-800">
                                        <?php echo number_format($holding_units, 4); ?>
                                    </span>
                                </div>
                                <div class="flex justify-between">
                                    <span class="text-xs text-blue-700">Value:</span>
                                    <span class="text-xs font-medium text-blue-800">
                                        $<?php echo number_format($holding_value, 2); ?>
                                    </span>
                                </div>
                            </div>
                        </div>

                        <div class="space-y-3">
                            <a href="investment_buy.php?id=<?php echo $plan_id; ?>"
                               class="w-full px-4 py-3 bg-black text-white text-sm font-medium rounded-lg hover:bg-gray-800 transition-colors duration-200 flex items-center justify-center">
                                <i data-lucide="plus" class="w-4 h-4 mr-2"></i> Buy More
                            </a>
                            <a href="investment_sell.php?id=<?php echo $plan_id; ?>"
                               class="w-full px-4 py-3 border border-gray-300 text-black text-sm font-medium rounded-lg hover:bg-gray-50 transition-colors duration-200 flex items-center justify-center">
                                <i data-lucide="minus" class="w-4 h-4 mr-2"></i> Sell Units
                            </a>
                        </div>
                    </div>

                    <!-- Plan Details -->
                    <div class="bg-white rounded-xl p-6 shadow-sm border border-gray-100">
                        <h3 class="text-lg font-light text-black mb-4">Plan Details</h3>
                        <div class="space-y-3">
                            <div class="flex justify-between">
                                <span class="text-xs text-gray-500">Type:</span>
                                <span class="text-xs font-medium text-black"><?php echo htmlspecialchars($plan_category); ?></span>
                            </div>
                            <div class="flex justify-between">
                                <span class="text-xs text-gray-500">Category:</span>
                                <span class="text-xs font-medium text-black"><?php echo htmlspecialchars($plan_category); ?></span>
                            </div>
                            <div class="flex justify-between">
                                <span class="text-xs text-gray-500">Risk Level:</span>
                                <span class="text-xs font-medium text-black"><?php echo ucfirst($plan_risk); ?></span>
                            </div>
                            <div class="flex justify-between">
                                <span class="text-xs text-gray-500">Expense Ratio:</span>
                                <span class="text-xs font-medium text-black">0.85%</span>
                            </div>
                            <div class="flex justify-between">
                                <span class="text-xs text-gray-500">Total Assets:</span>
                                <span class="text-xs font-medium text-black">$2,500,000</span>
                            </div>
                            <div class="flex justify-between">
                                <span class="text-xs text-gray-500">Inception:</span>
                                <span class="text-xs font-medium text-black">Jan 2020</span>
                            </div>
                        </div>
                    </div>

                    <!-- Recent Activity -->
                    <div class="bg-white rounded-xl p-6 shadow-sm border border-gray-100">
                        <h3 class="text-lg font-light text-black mb-4">Recent Activity</h3>
                        <div class="space-y-3">
                            <?php if (empty($activity)): ?>
                                <p class="text-xs text-gray-500 text-center">No activity yet.</p>
                            <?php else: ?>
                                <?php foreach ($activity as $act): ?>
                                    <div class="flex items-center space-x-3 p-2 bg-gray-50 rounded-lg">
                                        <div class="w-8 h-8 flex items-center justify-center">
                                            <i data-lucide="<?php echo $act['type'] === 'investment' ? 'plus' : 'minus'; ?>"
                                               class="w-4 h-4 <?php echo $act['type'] === 'investment' ? 'text-green-500' : 'text-red-500'; ?>"></i>
                                        </div>
                                        <div class="flex-1 min-w-0">
                                            <p class="text-xs font-medium text-black">
                                                <?php echo $act['type'] === 'investment' ? 'Buy' : 'Sell'; ?>
                                            </p>
                                            <p class="text-xs text-gray-500">
                                                <?php echo date('M j, Y', strtotime($act['created_at'])); ?>
                                            </p>
                                        </div>
                                        <div class="text-right">
                                            <p class="text-xs font-medium text-black">
                                                $<?php echo number_format($act['total_cost'] ?? $act['amount'], 2); ?>
                                            </p>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </div>
                    </div>

                </div>
            </div>
        </div>

        <!-- Chart.js script -->
        <script>
            document.addEventListener('DOMContentLoaded', function () {
                const ctx = document.getElementById('investmentChart');
                if (!ctx) return;

                const nav = <?php echo $plan_nav; ?>;
                const labels = ['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec'];
                const baseData = Array.from({length: 12}, (_, i) => nav * (0.95 + i * 0.03));

                const chart = new Chart(ctx, {
                    type: 'line',
                    data: {
                        labels: labels,
                        datasets: [{
                            label: '<?php echo addslashes($plan_name); ?> NAV',
                            data: baseData,
                            borderColor: '#000000',
                            backgroundColor: 'rgba(0,0,0,0.1)',
                            borderWidth: 2,
                            fill: true,
                            tension: 0.4,
                            pointRadius: 0,
                            pointHoverRadius: 6,
                            pointHoverBackgroundColor: '#000000',
                            pointHoverBorderColor: '#ffffff',
                            pointHoverBorderWidth: 2
                        }]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        interaction: { intersect: false, mode: 'index' },
                        plugins: {
                            legend: { display: false },
                            tooltip: {
                                backgroundColor: 'rgba(0,0,0,0.8)',
                                titleColor: '#fff',
                                bodyColor: '#fff',
                                borderColor: '#000',
                                borderWidth: 1,
                                cornerRadius: 8,
                                displayColors: false,
                                callbacks: { label: ctx => '$' + ctx.parsed.y.toFixed(2) }
                            }
                        },
                        scales: {
                            x: { grid: { display: false }, ticks: { color: '#6b7280', font: { size: 11 } } },
                            y: {
                                position: 'right',
                                grid: { color: 'rgba(0,0,0,0.1)', drawBorder: false },
                                ticks: {
                                    color: '#6b7280',
                                    font: { size: 11 },
                                    callback: v => '$' + Number(v).toFixed(2)
                                }
                            }
                        }
                    }
                });

                // Period buttons
                document.querySelectorAll('.period-btn').forEach(btn => {
                    btn.addEventListener('click', function () {
                        const period = this.dataset.period;
                        document.querySelectorAll('.period-btn').forEach(b => {
                            b.classList.remove('bg-black','text-white');
                            b.classList.add('bg-gray-100','text-gray-600');
                        });
                        this.classList.remove('bg-gray-100','text-gray-600');
                        this.classList.add('bg-black','text-white');

                        // Simulate different periods
                        const factor = period === '3y' ? 1.1 : (period === '5y' ? 1.2 : 1);
                        chart.data.datasets[0].data = baseData.map(v => v * factor);
                        chart.update();
                    });
                });
            });
        </script>
    </main>
</div>

<?php include 'footer.php'; ?>
</body>
</html>