<?php
include 'header.php';

/* -------------------------------------------------
   1. PORTFOLIO SUMMARY (from `investments` table)
   ------------------------------------------------- */
// Total invested: sum of amount from buy transactions
// Total sold: sum of amount from sell transactions
// Current value: remaining units * current plan_nav
$summary_sql = "
    SELECT
        COALESCE(SUM(CASE WHEN type = 'buy' THEN amount ELSE 0 END), 0) AS total_invested,
        COALESCE(SUM(CASE WHEN type = 'sell' THEN amount ELSE 0 END), 0) AS total_sold,
        COALESCE(SUM(CASE WHEN type = 'buy' THEN units ELSE -units END), 0) AS remaining_units
    FROM investments
    WHERE user_id = ?
";
$summary_stmt = $link->prepare($summary_sql);
$summary_stmt->bind_param('i', $user_id);
$summary_stmt->execute();
$summary = $summary_stmt->get_result()->fetch_assoc();

$total_invested = (float)$summary['total_invested'];
$total_sold = (float)$summary['total_sold'];
$remaining_units_total = (float)$summary['remaining_units']; // not used directly here, but accurate

// Current investment value: recalculate using current NAV per plan (done properly in holdings below)
$invest_value = 0; // will be updated from holdings query

$net_invested = $total_invested - $total_sold;
$gain_loss = 0; // will be updated after holdings
$gain_pct = 0;

/* -------------------------------------------------
   2. STOCK PORTFOLIO VALUE (unchanged)
   ------------------------------------------------- */
$stock_value_sql = "
    SELECT COALESCE(SUM(shares * price_per_share), 0) AS stock_value
    FROM stock_transactions
    WHERE user_id = ? AND type = 'buy'
";
$stock_stmt = $link->prepare($stock_value_sql);
$stock_stmt->bind_param('i', $user_id);
$stock_stmt->execute();
$stock_value = (float)$stock_stmt->get_result()->fetch_assoc()['stock_value'];

/* -------------------------------------------------
   3. INVESTMENT HOLDINGS (grouped by plan – from `investments`)
      This gives accurate current value using latest plan_nav
   ------------------------------------------------- */
$holdings_sql = "
    SELECT
        plan_name,
        SUM(units) AS total_units,
        plan_nav,
        (SUM(units) * plan_nav) AS current_value,
        SUM(CASE WHEN type = 'buy' THEN amount ELSE 0 END) AS cost_basis,
        (SUM(units) * plan_nav - SUM(CASE WHEN type = 'buy' THEN amount ELSE 0 END)) AS plan_gain,
        CASE 
            WHEN SUM(CASE WHEN type = 'buy' THEN amount ELSE 0 END) > 0
            THEN ((SUM(units) * plan_nav - SUM(CASE WHEN type = 'buy' THEN amount ELSE 0 END)) 
                  / SUM(CASE WHEN type = 'buy' THEN amount ELSE 0 END)) * 100
            ELSE 0 
        END AS gain_pct
    FROM investments
    WHERE user_id = ?
    GROUP BY plan_name, plan_nav
    HAVING total_units > 0
    ORDER BY current_value DESC
";
$holdings_stmt = $link->prepare($holdings_sql);
$holdings_stmt->bind_param('i', $user_id);
$holdings_stmt->execute();
$holdings_res = $holdings_stmt->get_result();

// Calculate total current investment value from holdings
$invest_value = 0;
while ($row = $holdings_res->fetch_assoc()) {
    $invest_value += (float)$row['current_value'];
}
$holdings_res->data_seek(0); // reset pointer for display loop

// Now calculate overall gain/loss
$gain_loss = $invest_value - $net_invested;
$gain_pct = $net_invested > 0 ? ($gain_loss / $net_invested) * 100 : 0;

/* -------------------------------------------------
   4. TOTAL PORTFOLIO VALUE
   ------------------------------------------------- */
$portfolio_value = $invest_value + $stock_value;
$total_gain_loss = $portfolio_value - $net_invested;
$total_gain_pct = $net_invested > 0 ? ($total_gain_loss / $net_invested) * 100 : 0;

/* -------------------------------------------------
   5. STOCK HOLDINGS (unchanged)
   ------------------------------------------------- */
$stock_holdings_sql = "
    SELECT
        symbol,
        SUM(CASE WHEN type = 'buy' THEN shares ELSE -shares END) AS total_shares,
        price_per_share,
        (SUM(CASE WHEN type = 'buy' THEN shares ELSE -shares END) * price_per_share) AS current_value,
        (
            (SUM(CASE WHEN type = 'buy' THEN shares ELSE -shares END) * price_per_share)
            - SUM(CASE WHEN type = 'buy' THEN total_amount ELSE -total_amount END)
        ) AS gain,
        CASE
            WHEN SUM(CASE WHEN type = 'buy' THEN total_amount ELSE -total_amount END) > 0
            THEN (
                (
                    (SUM(CASE WHEN type = 'buy' THEN shares ELSE -shares END) * price_per_share)
                    - SUM(CASE WHEN type = 'buy' THEN total_amount ELSE -total_amount END)
                )
                / ABS(SUM(CASE WHEN type = 'buy' THEN total_amount ELSE -total_amount END))
            ) * 100
            ELSE 0
        END AS gain_pct
    FROM stock_transactions
    WHERE user_id = ?
    GROUP BY symbol, price_per_share
    HAVING total_shares > 0
    ORDER BY current_value DESC
";
$stock_holdings_stmt = $link->prepare($stock_holdings_sql);
$stock_holdings_stmt->bind_param('i', $user_id);
$stock_holdings_stmt->execute();
$stock_holdings_res = $stock_holdings_stmt->get_result();

/* -------------------------------------------------
   6. RECENT INVESTMENT TRANSACTIONS (from `investments`)
   ------------------------------------------------- */
$invest_sql = "
    SELECT plan_name, amount, type, created_at
    FROM investments
    WHERE user_id = ? AND type IN ('buy','sell')
    ORDER BY created_at DESC LIMIT 5
";
$invest_stmt = $link->prepare($invest_sql);
$invest_stmt->bind_param('i', $user_id);
$invest_stmt->execute();
$invest_res = $invest_stmt->get_result();

/* -------------------------------------------------
   7. RECENT STOCK TRANSACTIONS (unchanged)
   ------------------------------------------------- */
$stock_tx_sql = "
    SELECT symbol, total_amount, type, created_at
    FROM stock_transactions
    WHERE user_id = ?
    ORDER BY created_at DESC LIMIT 5
";
$stock_tx_stmt = $link->prepare($stock_tx_sql);
$stock_tx_stmt->bind_param('i', $user_id);
$stock_tx_stmt->execute();
$stock_tx_res = $stock_tx_stmt->get_result();
?>
<!-- ====================== HTML ====================== -->
<div class="flex-1 flex flex-col overflow-hidden lg:ml-72">
    <!-- Top Bar -->
    <header class="bg-white border-b border-gray-200">
        <div class="flex items-center justify-between px-6 py-3">
            <div class="flex items-center">
                <button onclick="toggleSidebar()" class="lg:hidden text-gray-500 hover:text-gray-700 p-1.5 rounded-md hover:bg-gray-100 mr-3">
                    <i data-lucide="menu" class="w-5 h-5"></i>
                </button>
                <div class="font-light text-lg text-black">Investment Dashboard</div>
            </div>
        </div>
    </header>

    <!-- Main Content -->
    <main class="flex-1 overflow-y-auto p-6 bg-white pb-20 lg:pb-6">
        <div class="max-w-7xl mx-auto">
            <!-- HERO CARD -->
            <div class="bg-gradient-to-br from-black via-gray-900 to-gray-800 rounded-2xl p-6 mb-6 text-white relative overflow-hidden">
                <div class="absolute inset-0 opacity-10">
                    <div class="absolute top-0 right-0 w-48 h-48 bg-white rounded-full -translate-y-24 translate-x-24"></div>
                    <div class="absolute bottom-0 left-0 w-24 h-24 bg-white rounded-full translate-y-12 -translate-x-12"></div>
                </div>
                <div class="relative z-10">
                    <div class="flex flex-col lg:flex-row lg:items-center lg:justify-between">
                        <div class="mb-4 lg:mb-0 lg:flex-1">
                            <h1 class="text-xl font-light mb-1">Investment Dashboard</h1>
                            <p class="text-gray-300 text-sm">Complete overview of your investment portfolio</p>
                        </div>
                        <div class="bg-white bg-opacity-15 backdrop-blur-xl rounded-xl p-4 border border-white border-opacity-20 shadow-xl lg:w-80">
                            <div class="flex items-center justify-between mb-3">
                                <div>
                                    <p class="text-xs text-gray-300 mb-1">Total Portfolio</p>
                                    <p class="text-lg font-light">$<?= number_format($portfolio_value, 2) ?></p>
                                </div>
                                <div class="w-10 h-10 flex items-center justify-center">
                                    <i data-lucide="trending-up" class="w-5 h-5 text-white"></i>
                                </div>
                            </div>
                            <div class="flex space-x-4 text-xs">
                                <div class="flex-1 text-center">
                                    <p class="text-gray-300">Net Invested</p>
                                    <p class="text-white font-medium">$<?= number_format($net_invested, 0) ?></p>
                                </div>
                                <div class="flex-1 text-center">
                                    <p class="text-gray-300">Gain/Loss</p>
                                    <p class="text-white font-medium <?= $total_gain_loss >= 0 ? 'text-green-400' : 'text-red-400' ?>">
                                        <?= $total_gain_loss >= 0 ? '+' : '' ?>$<?= number_format(abs($total_gain_loss), 0) ?>
                                        (<?= $total_gain_loss >= 0 ? '+' : '' ?><?= number_format($total_gain_pct, 2) ?>%)
                                    </p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- PORTFOLIO OVERVIEW -->
            <div class="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
                <div class="bg-white rounded-xl p-4 shadow-sm border border-gray-100">
                    <div class="flex items-center justify-between mb-3">
                        <div>
                            <p class="text-xs text-gray-500 mb-1">Portfolio Value</p>
                            <p class="text-lg font-light text-black">$<?= number_format($portfolio_value, 2) ?></p>
                        </div>
                        <div class="w-8 h-8 flex items-center justify-center">
                            <i data-lucide="pie-chart" class="w-4 h-4 text-blue-600"></i>
                        </div>
                    </div>
                </div>
                <div class="bg-white rounded-xl p-4 shadow-sm border border-gray-100">
                    <div class="flex items-center justify-between mb-3">
                        <div>
                            <p class="text-xs text-gray-500 mb-1">Net Invested</p>
                            <p class="text-lg font-light text-black">$<?= number_format($net_invested, 2) ?></p>
                        </div>
                        <div class="w-8 h-8 flex items-center justify-center">
                            <i data-lucide="dollar-sign" class="w-4 h-4 text-green-600"></i>
                        </div>
                    </div>
                </div>
                <div class="bg-white rounded-xl p-4 shadow-sm border border-gray-100">
                    <div class="flex items-center justify-between mb-3">
                        <div>
                            <p class="text-xs text-gray-500 mb-1">Total Gain/Loss</p>
                            <p class="text-lg font-light <?= $total_gain_loss >= 0 ? 'text-green-600' : 'text-red-600' ?>">
                                <?= $total_gain_loss >= 0 ? '+' : '' ?>$<?= number_format(abs($total_gain_loss), 2) ?>
                            </p>
                            <p class="text-xs <?= $total_gain_loss >= 0 ? 'text-green-600' : 'text-red-600' ?>">
                                <?= $total_gain_loss >= 0 ? '+' : '' ?><?= number_format($total_gain_pct, 2) ?>%
                            </p>
                        </div>
                        <div class="w-8 h-8 flex items-center justify-center">
                            <i data-lucide="arrow-up" class="w-4 h-4 <?= $total_gain_loss >= 0 ? 'text-green-600' : 'text-red-600' ?>"></i>
                        </div>
                    </div>
                </div>
                <div class="bg-white rounded-xl p-4 shadow-sm border border-gray-100">
                    <div class="flex items-center justify-between mb-3">
                        <div>
                            <p class="text-xs text-gray-500 mb-1">Monthly Performance</p>
                            <p class="text-lg font-light text-green-600">+5.26%</p>
                            <p class="text-xs text-gray-500">This month</p>
                        </div>
                        <div class="w-8 h-8 flex items-center justify-center">
                            <i data-lucide="trending-up" class="w-4 h-4 text-green-600"></i>
                        </div>
                    </div>
                </div>
            </div>

            <div class="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-6">
                <!-- Investment Holdings -->
                <div class="bg-white rounded-xl p-6 shadow-sm border border-gray-100">
                    <div class="flex items-center justify-between mb-4">
                        <div>
                            <h3 class="text-lg font-light text-black mb-1">Investment Holdings</h3>
                            <p class="text-xs text-gray-500"><?= $holdings_res->num_rows ?> active plans</p>
                        </div>
                        <a href="holdings.php" class="text-xs text-blue-600 hover:text-blue-800 font-medium">
                            View All <i data-lucide="arrow-right" class="w-3 h-3 inline ml-1"></i>
                        </a>
                    </div>
                    <div class="space-y-3">
                        <?php if ($holdings_res->num_rows > 0): ?>
                            <?php while ($row = $holdings_res->fetch_assoc()): ?>
                                <div class="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                                    <div class="flex items-center space-x-3">
                                        <div class="w-8 h-8 bg-gray-100 rounded-lg flex items-center justify-center">
                                            <i data-lucide="trending-up" class="w-4 h-4 text-gray-600"></i>
                                        </div>
                                        <div>
                                            <p class="text-sm font-medium text-black"><?= htmlspecialchars($row['plan_name']) ?></p>
                                            <p class="text-xs text-gray-500"><?= number_format($row['total_units'], 4) ?> units @ $<?= number_format($row['plan_nav'], 4) ?>/unit</p>
                                        </div>
                                    </div>
                                    <div class="text-right">
                                        <p class="text-sm font-medium text-black">$<?= number_format($row['current_value'], 2) ?></p>
                                        <p class="text-xs <?= $row['gain_pct'] >= 0 ? 'text-green-600' : 'text-red-600' ?>">
                                            <?= $row['gain_pct'] >= 0 ? '+' : '' ?><?= number_format($row['gain_pct'], 2) ?>%
                                        </p>
                                    </div>
                                </div>
                            <?php endwhile; ?>
                        <?php else: ?>
                            <p class="text-xs text-gray-500 text-center py-4">No active investment holdings.</p>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Stock Holdings -->
                <div class="bg-white rounded-xl p-6 shadow-sm border border-gray-100">
                    <div class="flex items-center justify-between mb-4">
                        <div>
                            <h3 class="text-lg font-light text-black mb-1">Stock Holdings</h3>
                            <p class="text-xs text-gray-500"><?= $stock_holdings_res->num_rows ?> stocks</p>
                        </div>
                        <a href="portfolio.php" class="text-xs text-blue-600 hover:text-blue-800 font-medium">
                            View All <i data-lucide="arrow-right" class="w-3 h-3 inline ml-1"></i>
                        </a>
                    </div>
                    <div class="space-y-3">
                        <?php if ($stock_holdings_res->num_rows > 0): ?>
                            <?php while ($row = $stock_holdings_res->fetch_assoc()): ?>
                                <div class="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                                    <div class="flex items-center space-x-3">
                                        <div class="w-8 h-8 bg-gray-100 rounded-lg flex items-center justify-center">
                                            <i data-lucide="bar-chart-3" class="w-4 h-4 text-gray-600"></i>
                                        </div>
                                        <div>
                                            <p class="text-sm font-medium text-black"><?= htmlspecialchars($row['symbol']) ?></p>
                                            <p class="text-xs text-gray-500"><?= number_format($row['total_shares'], 4) ?> shares</p>
                                        </div>
                                    </div>
                                    <div class="text-right">
                                        <p class="text-sm font-medium text-black">$<?= number_format($row['current_value'], 2) ?></p>
                                        <p class="text-xs <?= $row['gain_pct'] >= 0 ? 'text-green-600' : 'text-red-600' ?>">
                                            <?= $row['gain_pct'] >= 0 ? '+' : '' ?><?= number_format($row['gain_pct'], 2) ?>%
                                        </p>
                                    </div>
                                </div>
                            <?php endwhile; ?>
                        <?php else: ?>
                            <p class="text-xs text-gray-500 text-center py-4">No stock holdings.</p>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <!-- RECENT ACTIVITY -->
            <div class="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-6">
                <!-- Recent Investment Transactions -->
                <div class="bg-white rounded-xl p-6 shadow-sm border border-gray-100">
                    <div class="flex items-center justify-between mb-4">
                        <div>
                            <h3 class="text-lg font-light text-black mb-1">Recent Investment Activity</h3>
                            <p class="text-xs text-gray-500">Latest fund purchases and redemptions</p>
                        </div>
                        <a href="transactions.php" class="text-xs text-blue-600 hover:text-blue-800 font-medium">
                            View All <i data-lucide="arrow-right" class="w-3 h-3 inline ml-1"></i>
                        </a>
                    </div>
                    <div class="space-y-3">
                        <?php if ($invest_res->num_rows > 0): ?>
                            <?php while ($row = $invest_res->fetch_assoc()): ?>
                                <div class="flex items-center space-x-3 p-3 bg-gray-50 rounded-lg">
                                    <div class="w-8 h-8 flex items-center justify-center">
                                        <?php if (strtolower($row['type']) === 'buy'): ?>
                                            <i data-lucide="plus" class="w-4 h-4 text-green-500"></i>
                                        <?php else: ?>
                                            <i data-lucide="minus" class="w-4 h-4 text-red-500"></i>
                                        <?php endif; ?>
                                    </div>
                                    <div class="flex-1 min-w-0">
                                        <p class="text-xs font-medium text-black"><?= htmlspecialchars($row['plan_name']) ?></p>
                                        <p class="text-xs text-gray-500">
                                            <?= ucfirst($row['type']) ?> • <?= date('M j, Y', strtotime($row['created_at'])) ?>
                                        </p>
                                    </div>
                                    <div class="text-right">
                                        <p class="text-xs font-medium text-black">$<?= number_format($row['amount'], 2) ?></p>
                                    </div>
                                </div>
                            <?php endwhile; ?>
                        <?php else: ?>
                            <p class="text-xs text-gray-500 text-center py-4">No recent investment activity.</p>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Recent Stock Transactions -->
                <div class="bg-white rounded-xl p-6 shadow-sm border border-gray-100">
                    <div class="flex items-center justify-between mb-4">
                        <div>
                            <h3 class="text-lg font-light text-black mb-1">Recent Stock Activity</h3>
                            <p class="text-xs text-gray-500">Latest stock trades</p>
                        </div>
                        <a href="trading_transactions.php" class="text-xs text-blue-600 hover:text-blue-800 font-medium">
                            View All <i data-lucide="arrow-right" class="w-3 h-3 inline ml-1"></i>
                        </a>
                    </div>
                    <div class="space-y-3">
                        <?php if ($stock_tx_res->num_rows > 0): ?>
                            <?php while ($row = $stock_tx_res->fetch_assoc()): ?>
                                <div class="flex items-center space-x-3 p-3 bg-gray-50 rounded-lg">
                                    <div class="w-8 h-8 flex items-center justify-center">
                                        <?php if (strtolower($row['type']) === 'buy'): ?>
                                            <i data-lucide="plus" class="w-4 h-4 text-green-500"></i>
                                        <?php else: ?>
                                            <i data-lucide="minus" class="w-4 h-4 text-red-500"></i>
                                        <?php endif; ?>
                                    </div>
                                    <div class="flex-1 min-w-0">
                                        <p class="text-xs font-medium text-black"><?= htmlspecialchars($row['symbol']) ?></p>
                                        <p class="text-xs text-gray-500">
                                            <?= ucfirst($row['type']) ?> • <?= date('M j, Y', strtotime($row['created_at'])) ?>
                                        </p>
                                    </div>
                                    <div class="text-right">
                                        <p class="text-xs font-medium text-black">$<?= number_format($row['total_amount'], 2) ?></p>
                                    </div>
                                </div>
                            <?php endwhile; ?>
                        <?php else: ?>
                            <p class="text-xs text-gray-500 text-center py-4">No recent stock activity.</p>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </main>
</div>

<?php include 'footer.php'; ?>