<?php
include 'header.php';

// -------------------------------------------------
// 2. FETCH ALL RELEVANT TRANSACTIONS FROM investments TABLE
// -------------------------------------------------
$sql = "
    SELECT
        i.plan_id,
        i.plan_name,
        i.plan_category,
        i.plan_risk,
        i.plan_nav,
        i.amount,
        i.units,
        i.type,
        i.fee,
        i.total_cost
    FROM investments i
    WHERE i.user_id = ?
      AND i.type IN ('buy','sell')
    ORDER BY i.plan_id, i.created_at
";

$stmt = mysqli_prepare($link, $sql);
if (!$stmt) {
    die("Prepare failed: " . mysqli_error($link));
}
mysqli_stmt_bind_param($stmt, "i", $user_id);
mysqli_stmt_execute($stmt);
$res = mysqli_stmt_get_result($stmt);

// -------------------------------------------------
// 3. BUILD AN ARRAY WITH CURRENT HOLDINGS
// -------------------------------------------------
$holdings = [];
while ($row = mysqli_fetch_assoc($res)) {
    $pid = $row['plan_id'] ?? 'unknown';

    if (!isset($holdings[$pid])) {
        $holdings[$pid] = [
            'name' => $row['plan_name'] ?? 'Unnamed Plan',
            'category' => $row['plan_category'] ?? '',
            'risk' => $row['plan_risk'] ?? '',
            'nav' => (float)($row['plan_nav'] ?? 0),
            'units' => 0.0,
            'invested' => 0.0,
            'avg_cost' => 0.0,
        ];
    }

    $units = (float)($row['units'] ?? 0);
    $cost = (float)($row['total_cost'] ?? 0);

    if ($row['type'] === 'sell') {
        $units = -$units;
        $cost = -$cost;
    }

    $holdings[$pid]['units'] += $units;
    $holdings[$pid]['invested'] += $cost;
}

// Recalculate average cost for each plan
foreach ($holdings as &$h) {
    if ($h['units'] > 0) {
        $h['avg_cost'] = $h['invested'] / $h['units'];
    } else {
        $h['avg_cost'] = 0;
    }
}
unset($h); // break reference

// -------------------------------------------------
// 4. CALCULATE TOTALS FOR THE SUMMARY CARD
// -------------------------------------------------
$totalInvested = 0;
$totalValue = 0;
$totalGain = 0;

foreach ($holdings as $h) {
    if ($h['units'] <= 0) continue; // skip sold-out or negative holdings

    $value = $h['units'] * $h['nav'];
    $totalInvested += $h['invested'];
    $totalValue += $value;
    $totalGain += ($value - $h['invested']);
}

$totalReturnPct = ($totalInvested > 0) ? ($totalGain / $totalInvested) * 100 : 0;

mysqli_close($link);
?>
<div class="flex-1 flex flex-col overflow-hidden lg:ml-72">
    <!-- Top Bar -->
    <header class="bg-white border-b border-gray-200">
        <div class="flex items-center justify-between px-6 py-3">
            <div class="flex items-center">
                <button onclick="toggleSidebar()" class="lg:hidden text-gray-500 hover:text-gray-700 p-1.5 rounded-md hover:bg-gray-100 mr-3">
                    <i data-lucide="menu" class="w-5 h-5"></i>
                </button>
                <div class="font-light text-lg text-black">Investment Holdings</div>
            </div>
        </div>
    </header>

    <!-- ============================================= -->
    <!-- PAGE CONTENT -->
    <!-- ============================================= -->
    <main class="flex-1 overflow-y-auto p-6 bg-white pb-20 lg:pb-6">
        <div class="max-w-7xl mx-auto">
            <!-- Enhanced Header -->
            <div class="bg-gradient-to-br from-black via-gray-900 to-gray-800 rounded-2xl p-6 mb-6 text-white relative overflow-hidden">
                <div class="absolute inset-0 opacity-10">
                    <div class="absolute top-0 right-0 w-48 h-48 bg-white rounded-full -translate-y-24 translate-x-24"></div>
                    <div class="absolute bottom-0 left-0 w-24 h-24 bg-white rounded-full translate-y-12 -translate-x-12"></div>
                </div>
                <div class="relative z-10">
                    <div class="flex flex-col lg:flex-row lg:items-center lg:justify-between">
                        <div class="mb-4 lg:mb-0 lg:flex-1">
                            <h1 class="text-xl font-light mb-1">Your Holdings</h1>
                            <p class="text-gray-300 text-sm">Detailed view of your active investment positions</p>
                        </div>
                        <div class="bg-white bg-opacity-15 backdrop-blur-xl rounded-xl p-4 border border-white border-opacity-20 shadow-xl lg:w-80">
                            <div class="flex items-center justify-between mb-3">
                                <div>
                                    <p class="text-xs text-gray-300 mb-1">Active Holdings</p>
                                    <p class="text-lg font-light"><?= count(array_filter($holdings, fn($h) => $h['units'] > 0)) ?></p>
                                </div>
                                <div class="w-10 h-10 flex items-center justify-center">
                                    <i data-lucide="pie-chart" class="w-5 h-5 text-white"></i>
                                </div>
                            </div>
                            <div class="flex space-x-4 text-xs">
                                <div class="flex-1 text-center">
                                    <p class="text-gray-300">Current Value</p>
                                    <p class="text-white font-medium">$<?= number_format($totalValue, 2) ?></p>
                                </div>
                                <div class="flex-1 text-center">
                                    <p class="text-gray-300">Gain/Loss</p>
                                    <p class="text-white font-medium <?= $totalGain >= 0 ? 'text-green-400' : 'text-red-400' ?>">
                                        <?= $totalGain >= 0 ? '+' : '' ?>$<?= number_format($totalGain, 2) ?>
                                        (<?= $totalGain >= 0 ? '+' : '' ?><?= number_format($totalReturnPct, 2) ?>%)
                                    </p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Holdings Table -->
            <div class="bg-white rounded-xl shadow-sm border border-gray-100 overflow-hidden">
                <div class="px-6 py-4 border-b border-gray-100">
                    <div class="flex items-center justify-between">
                        <div>
                            <h3 class="text-lg font-light text-black mb-1">Investment Holdings</h3>
                            <p class="text-xs text-gray-500">Detailed breakdown of your active positions</p>
                        </div>
                        <a href="investments.php"
                           class="text-xs text-blue-600 hover:text-blue-800 transition-colors duration-200 font-medium">
                            View Transactions <i data-lucide="arrow-right" class="w-3 h-3 inline ml-1"></i>
                        </a>
                    </div>
                </div>
                <div class="overflow-x-auto">
                    <table class="w-full">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Investment</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Units</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Avg Cost</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Current NAV</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Current Value</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Gain/Loss</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Return %</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                            </tr>
                        </thead>
                        <tbody class="bg-white divide-y divide-gray-100">
                            <?php if (empty($holdings)): ?>
                                <tr>
                                    <td colspan="8" class="px-6 py-8 text-center text-gray-500">No investment holdings found.</td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($holdings as $pid => $h): ?>
                                    <?php if ($h['units'] <= 0) continue; // Only show active holdings ?>
                                    <?php
                                        $currValue = $h['units'] * $h['nav'];
                                        $gainLoss = $currValue - $h['invested'];
                                        $pct = $h['invested'] > 0 ? ($gainLoss / $h['invested']) * 100 : 0;
                                    ?>
                                    <tr class="hover:bg-gray-50 transition-colors duration-200">
                                        <td class="px-6 py-4">
                                            <div class="flex items-center">
                                                <div class="flex-shrink-0 w-8 h-8 bg-gray-100 rounded-lg flex items-center justify-center">
                                                    <i data-lucide="trending-up" class="w-4 h-4 text-gray-600"></i>
                                                </div>
                                                <div class="ml-3">
                                                    <div class="text-sm font-medium text-black"><?= htmlspecialchars($h['name']) ?></div>
                                                    <div class="text-xs text-gray-500"><?= htmlspecialchars($h['category']) ?> • <?= ucfirst($h['risk']) ?></div>
                                                </div>
                                            </div>
                                        </td>
                                        <td class="px-6 py-4"><div class="text-sm text-black"><?= number_format($h['units'], 4) ?></div></td>
                                        <td class="px-6 py-4"><div class="text-sm text-black">$<?= number_format($h['avg_cost'], 4) ?></div></td>
                                        <td class="px-6 py-4"><div class="text-sm text-black">$<?= number_format($h['nav'], 4) ?></div></td>
                                        <td class="px-6 py-4"><div class="text-sm font-medium text-black">$<?= number_format($currValue, 2) ?></div></td>
                                        <td class="px-6 py-4">
                                            <div class="text-sm font-medium <?= $gainLoss >= 0 ? 'text-green-600' : 'text-red-600' ?>">
                                                <?= $gainLoss >= 0 ? '+' : '' ?>$<?= number_format($gainLoss, 2) ?>
                                            </div>
                                        </td>
                                        <td class="px-6 py-4">
                                            <div class="text-sm font-medium <?= $pct >= 0 ? 'text-green-600' : 'text-red-600' ?>">
                                                <?= $pct >= 0 ? '+' : '' ?><?= number_format($pct, 2) ?>%
                                            </div>
                                        </td>
                                        <td class="px-6 py-4">
                                            <div class="flex space-x-2">
                                                <!-- BUY -->
                                                <a href="investment_buy.php?id=<?= $pid ?>"
                                                   class="w-8 h-8 flex items-center justify-center bg-green-100 text-green-700 hover:bg-green-200 rounded-lg transition-colors duration-200"
                                                   title="Buy More">
                                                    <i data-lucide="plus" class="w-4 h-4"></i>
                                                </a>
                                                <!-- SELL -->
                                                <a href="investment_sell.php?id=<?= $pid ?>"
                                                   class="w-8 h-8 flex items-center justify-center bg-red-100 text-red-700 hover:bg-red-200 rounded-lg transition-colors duration-200"
                                                   title="Sell">
                                                    <i data-lucide="minus" class="w-4 h-4"></i>
                                                </a>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- Summary Cards -->
            <div class="grid grid-cols-1 md:grid-cols-4 gap-4 mt-6">
                <div class="bg-white rounded-xl p-4 shadow-sm border border-gray-100">
                    <div class="flex items-center justify-between mb-3">
                        <div>
                            <p class="text-xs text-gray-500 mb-1">Total Invested</p>
                            <p class="text-lg font-light text-black">$<?= number_format($totalInvested, 2) ?></p>
                        </div>
                        <div class="w-8 h-8 flex items-center justify-center">
                            <i data-lucide="dollar-sign" class="w-4 h-4 text-blue-600"></i>
                        </div>
                    </div>
                </div>
                <div class="bg-white rounded-xl p-4 shadow-sm border border-gray-100">
                    <div class="flex items-center justify-between mb-3">
                        <div>
                            <p class="text-xs text-gray-500 mb-1">Current Value</p>
                            <p class="text-lg font-light text-black">$<?= number_format($totalValue, 2) ?></p>
                        </div>
                        <div class="w-8 h-8 flex items-center justify-center">
                            <i data-lucide="trending-up" class="w-4 h-4 text-green-600"></i>
                        </div>
                    </div>
                </div>
                <div class="bg-white rounded-xl p-4 shadow-sm border border-gray-100">
                    <div class="flex items-center justify-between mb-3">
                        <div>
                            <p class="text-xs text-gray-500 mb-1">Total Gain/Loss</p>
                            <p class="text-lg font-light <?= $totalGain >= 0 ? 'text-green-600' : 'text-red-600' ?>">
                                <?= $totalGain >= 0 ? '+' : '' ?>$<?= number_format($totalGain, 2) ?>
                            </p>
                        </div>
                        <div class="w-8 h-8 flex items-center justify-center">
                            <i data-lucide="arrow-up" class="w-4 h-4 <?= $totalGain >= 0 ? 'text-green-600' : 'text-red-600' ?>"></i>
                        </div>
                    </div>
                </div>
                <div class="bg-white rounded-xl p-4 shadow-sm border border-gray-100">
                    <div class="flex items-center justify-between mb-3">
                        <div>
                            <p class="text-xs text-gray-500 mb-1">Overall Return</p>
                            <p class="text-lg font-light <?= $totalReturnPct >= 0 ? 'text-green-600' : 'text-red-600' ?>">
                                <?= $totalReturnPct >= 0 ? '+' : '' ?><?= number_format($totalReturnPct, 2) ?>%
                            </p>
                        </div>
                        <div class="w-8 h-8 flex items-center justify-center">
                            <i data-lucide="percent" class="w-4 h-4 <?= $totalReturnPct >= 0 ? 'text-green-600' : 'text-red-600' ?>"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </main>
</div>

<?php include 'footer.php'; ?>