<?php
include 'header.php';


// -------------------------------------------------
// 3. FETCH ORDERS + CARS
// -------------------------------------------------
$sql = "
    SELECT 
        o.id,
        o.car_id,
        o.amount,
        o.status,
        o.payment_method,
        o.order_date,
        o.created_at,
        o.down_payment,
        o.monthly_payment,
        o.term_months,
        c.title AS car_title,
        c.year,
        c.image_url
    FROM orders o
    LEFT JOIN cars c ON o.car_id = c.id
    WHERE o.user_id = ?
    ORDER BY o.order_date DESC, o.created_at DESC
";

$stmt = mysqli_prepare($link, $sql);
if (!$stmt) die("SQL Error: " . mysqli_error($link));

mysqli_stmt_bind_param($stmt, 'i', $user_id);
mysqli_stmt_execute($stmt);
$result = mysqli_stmt_get_result($stmt);

// -------------------------------------------------
// 4. BUILD PURCHASES + STATS
// -------------------------------------------------
$purchases = [];
$total = $completed = $pending = 0;

while ($row = mysqli_fetch_assoc($result)) {
    $purchases[] = $row;
    $total++;
    $status = strtolower($row['status']);
    if ($status === 'completed') $completed++;
    elseif ($status === 'pending') $pending++;
}

mysqli_close($link);

// -------------------------------------------------
// 5. FORMAT DATE/TIME IN WAT (Africa/Lagos)
// -------------------------------------------------
function formatWAT($datetime) {
    if (!$datetime) return ['date' => '', 'time' => ''];
    $dt = new DateTime($datetime, new DateTimeZone('UTC'));
    $dt->setTimezone(new DateTimeZone('Africa/Lagos'));
    return [
        'date' => $dt->format('M j, Y'),
        'time' => $dt->format('g:i A')
    ];
}
?>

<!-- ============================================= -->
<!--               PAGE CONTENT                    -->
<!-- ============================================= -->
<div class="flex-1 flex flex-col overflow-hidden lg:ml-72">
    <!-- Top Bar -->
    <header class="bg-white border-b border-gray-200">
        <div class="flex items-center justify-between px-6 py-3">
            <div class="flex items-center">
                <button onclick="toggleSidebar()" class="lg:hidden text-gray-500 hover:text-gray-700 p-1.5 rounded-md hover:bg-gray-100 mr-3">
                    <i data-lucide="menu" class="w-5 h-5"></i>
                </button>
                <div class="font-light text-lg text-black">Purchase History</div>
            </div>
           
        </div>
    </header>

    <!-- Main Content -->
    <main class="flex-1 overflow-y-auto p-6 bg-white pb-20 lg:pb-6">
        <div class="max-w-7xl mx-auto">

            <!-- Header Card -->
            <div class="bg-gradient-to-br from-black via-gray-900 to-gray-800 rounded-2xl p-6 mb-6 text-white relative overflow-hidden">
                <div class="absolute inset-0 opacity-10">
                    <div class="absolute top-0 right-0 w-48 h-48 bg-white rounded-full -translate-y-24 translate-x-24"></div>
                    <div class="absolute bottom-0 left-0 w-24 h-24 bg-white rounded-full translate-y-12 -translate-x-12"></div>
                </div>
                <div class="relative z-10">
                    <div class="flex flex-col lg:flex-row lg:items-center lg:justify-between">
                        <div class="mb-4 lg:mb-0 lg:flex-1">
                            <h1 class="text-xl font-light mb-1">Purchase History</h1>
                            <p class="text-gray-300 text-sm">Track all your vehicle purchases and transactions</p>
                        </div>
                        <div class="bg-white bg-opacity-15 backdrop-blur-xl rounded-xl p-4 border border-white border-opacity-20 shadow-xl lg:w-64">
                            <div class="flex items-center justify-between mb-3">
                                <div>
                                    <p class="text-xs text-gray-300 mb-1">Total Purchases</p>
                                    <p class="text-lg font-light"><?= $total ?></p>
                                </div>
                                <div class="w-10 h-10 flex items-center justify-center">
                                    <i data-lucide="receipt" class="w-5 h-5 text-white"></i>
                                </div>
                            </div>
                            <div class="flex space-x-4 text-xs">
                                <div class="flex-1 text-center">
                                    <p class="text-gray-300">Completed</p>
                                    <p class="text-white font-medium"><?= $completed ?></p>
                                </div>
                                <div class="flex-1 text-center">
                                    <p class="text-gray-300">Pending</p>
                                    <p class="text-white font-medium"><?= $pending ?></p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Table -->
            <div class="bg-white rounded-xl shadow-sm border border-gray-100">
                <div class="p-6 border-b border-gray-100">
                    <h2 class="text-lg font-semibold text-gray-900">All Purchases</h2>
                    <p class="text-sm text-gray-600 mt-1">Complete history of your vehicle purchases</p>
                </div>
                <div class="overflow-x-auto">
                    <table class="w-full">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Purchase</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Vehicle</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Amount</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Date</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                            </tr>
                        </thead>
                        <tbody class="bg-white divide-y divide-gray-200">
                            <?php if (empty($purchases)): ?>
                                <tr>
                                    <td colspan="6" class="px-6 py-8 text-center text-gray-500">No purchases found.</td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($purchases as $p): 
                                    $dt = formatWAT($p['order_date'] ?: $p['created_at']);
                                    $method = ucfirst($p['payment_method'] ?? 'Unknown');
                                    $status = strtolower($p['status']);
                                    $badge = $status === 'completed' ? 'bg-green-100 text-green-800' : 'bg-yellow-100 text-yellow-800';
                                    $label = ucfirst($status);
                                ?>
                                    <tr class="hover:bg-gray-50">
                                        <td class="px-6 py-4 whitespace-nowrap">
                                            <div class="flex items-center">
                                                <div class="w-10 h-10 bg-gradient-to-br from-blue-400 to-blue-600 rounded-lg flex items-center justify-center mr-3">
                                                    <i data-lucide="car" class="w-5 h-5 text-white"></i>
                                                </div>
                                                <div>
                                                    <div class="text-sm font-medium text-gray-900">#<?= $p['id'] ?></div>
                                                    <div class="text-sm text-gray-500"><?= htmlspecialchars($method) ?></div>
                                                </div>
                                            </div>
                                        </td>
                                        <td class="px-6 py-4 whitespace-nowrap">
                                            <div class="flex items-center">
                                                <div class="w-10 h-10 rounded-lg mr-3 overflow-hidden bg-gray-200">
                                                    <?php if ($p['image_url']): ?>
                                                        <img src="<?= htmlspecialchars($p['image_url']) ?>" alt="Car" class="w-full h-full object-cover">
                                                    <?php else: ?>
                                                        <div class="w-full h-full flex items-center justify-center">
                                                            <i data-lucide="car" class="w-5 h-5 text-gray-500"></i>
                                                        </div>
                                                    <?php endif; ?>
                                                </div>
                                                <div>
                                                    <div class="text-sm font-medium text-gray-900"><?= htmlspecialchars($p['car_title'] ?? 'Unknown') ?></div>
                                                    <div class="text-sm text-gray-500"><?= $p['year'] ?? '' ?> • <?= $p['color'] ?? '—' ?></div>
                                                </div>
                                            </div>
                                        </td>
                                        <td class="px-6 py-4 whitespace-nowrap">
                                            <div class="text-sm font-medium text-gray-900">$<?= number_format((float)$p['amount'], 2) ?></div>
                                        </td>
                                        <td class="px-6 py-4 whitespace-nowrap">
                                            <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium <?= $badge ?>">
                                                <?= $label ?>
                                            </span>
                                        </td>
                                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                            <?= $dt['date'] ?>
                                            <div class="text-xs text-gray-500"><?= $dt['time'] ?></div>
                                        </td>
                                        <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                            <div class="flex space-x-2">
                                             
                                                <button onclick='viewPurchase(<?= json_encode([
                                                    "id" => $p['id'],
                                                    "amount" => $p['amount'],
                                                    "status" => $p['status'],
                                                    "purchased_at" => $p['order_date'] ?: $p['created_at'],
                                                    "payment_method" => ["name" => $method],
                                                    "car" => [
                                                        "title" => $p['car_title'],
                                                        "year" => $p['year'],
                                                        "image_url" => $p['image_url']
                                                    ],
                                                    "down_payment" => $p['down_payment'],
                                                    "monthly_payment" => $p['monthly_payment'],
                                                    "term_months" => $p['term_months']
                                                ], JSON_UNESCAPED_SLASHES) ?>)'
                                                    class="w-8 h-8 flex items-center justify-center text-gray-600 hover:text-gray-800 hover:bg-gray-100 rounded-lg transition-colors duration-200"
                                                    title="View Details">
                                                    <i data-lucide="eye" class="w-4 h-4"></i>
                                                </button>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>

        <!-- Modal Script -->
        <script>
            function viewPurchase(p) {
                const car = p.car || {};
                const method = p.payment_method || {};
                const date = p.purchased_at ? new Date(p.purchased_at) : null;
                const fmt = (n) => `$${Number(n || 0).toLocaleString(undefined, {minimumFractionDigits:2, maximumFractionDigits:2})}`;
                const fmtDate = (d) => d ? d.toLocaleString('en-US', {month:'short', day:'2-digit', year:'numeric', hour:'numeric', minute:'2-digit', hour12:true}) : '';
                const badge = p.status.toLowerCase() === 'completed' ? 'bg-green-100 text-green-800' : 'bg-yellow-100 text-yellow-800';

                const lease = p.down_payment ? `
                    <div class="mt-6">
                        <h4 class="text-sm font-medium text-black mb-2">Lease Details</h4>
                        <div class="space-y-2 text-sm">
                            <div class="flex justify-between"><span class="text-gray-600">Down Payment</span><span class="font-medium">${fmt(p.down_payment)}</span></div>
                            <div class="flex justify-between"><span class="text-gray-600">Monthly</span><span class="font-medium">${fmt(p.monthly_payment)}</span></div>
                            <div class="flex justify-between"><span class="text-gray-600">Term</span><span class="font-medium">${p.term_months} months</span></div>
                        </div>
                    </div>` : '';

                const modal = document.createElement('div');
                modal.className = 'fixed inset-0 bg-black bg-opacity-40 flex items-center justify-center z-[1000] p-4';
                modal.innerHTML = `
                <div class="bg-white rounded-2xl shadow-2xl max-w-2xl w-full overflow-hidden">
                    <div class="bg-gradient-to-br from-black to-gray-900 text-white p-6">
                        <div class="flex justify-between items-start">
                            <div>
                                <div class="text-xs uppercase text-white/70">Purchase</div>
                                <div class="text-xl font-light">#${p.id}</div>
                            </div>
                            <span class="px-3 py-1 rounded-full text-xs font-medium ${badge}">${p.status.charAt(0).toUpperCase() + p.status.slice(1)}</span>
                        </div>
                    </div>
                    <div class="p-6">
                        <div class="grid md:grid-cols-2 gap-6">
                            <div>
                                <h4 class="font-medium mb-2">Details</h4>
                                <div class="text-sm space-y-1">
                                    <div class="flex justify-between"><span class="text-gray-600">Amount</span><span>${fmt(p.amount)}</span></div>
                                    <div class="flex justify-between"><span class="text-gray-600">Method</span><span>${method.name}</span></div>
                                    <div class="flex justify-between"><span class="text-gray-600">Date</span><span>${fmtDate(date)}</span></div>
                                </div>
                            </div>
                            <div>
                                <h4 class="font-medium mb-2">Vehicle</h4>
                                <div class="flex items-center">
                                    ${car.image_url ? `<img src="${car.image_url}" class="w-14 h-14 rounded-lg mr-3 object-cover">` : `<div class="w-14 h-14 bg-gray-200 rounded-lg mr-3 flex items-center justify-center"><i data-lucide="car" class="w-8 h-8 text-gray-500"></i></div>`}
                                    <div>
                                        <div class="font-medium">${car.title || '—'}</div>
                                        <div class="text-xs text-gray-600">${car.year || ''}</div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        ${lease}
                        
                    </div>
                </div>`;
                document.body.appendChild(modal);
                modal.addEventListener('click', (e) => { if (e.target === modal) modal.remove(); });
            }

           
        </script>
    </main>
</div>

<?php include 'footer.php'; ?>