<?php
include 'session.php';

// --------------------------------------------------
// 2. GET CAR ID
// --------------------------------------------------
$car_id = isset($_GET['id']) ? intval($_GET['id']) : 0;
if ($car_id <= 0) {
    echo '<script>alert("Invalid car ID."); window.location="cars.php";</script>';
    exit;
}

// --------------------------------------------------
// 3. FETCH CAR DETAILS
// --------------------------------------------------
$car_sql = "SELECT id, title, model, year, price, range_miles, accel_0_60, top_speed, image_url, status FROM cars WHERE id = ? LIMIT 1";
$car_stmt = $link->prepare($car_sql);
$car_stmt->bind_param('i', $car_id);
$car_stmt->execute();
$car_res = $car_stmt->get_result();

if ($car_res->num_rows === 0) {
    $car_stmt->close();
    echo '<script>alert("Car not found."); window.location="cars.php";</script>';
    exit;
}
$car = $car_res->fetch_assoc();
$car_stmt->close();

// --------------------------------------------------
// 4. PROCESS ORDER (if form submitted)
// --------------------------------------------------
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['place_order'])) {
    // Re-validate car is still available
    $check_sql = "SELECT status, price FROM cars WHERE id = ? LIMIT 1";
    $check_stmt = $link->prepare($check_sql);
    $check_stmt->bind_param('i', $car_id);
    $check_stmt->execute();
    $check_res = $check_stmt->get_result();
    $current_car = $check_res->fetch_assoc();
    $check_stmt->close();

    if (!$current_car || $current_car['status'] !== 'available') {
        echo '<script>alert("This vehicle is no longer available."); window.location="car_details.php?id=' . $car_id . '";</script>';
        exit;
    }

    $car_price = (float)$current_car['price'];
    $payment_type = $_POST['payment_type'] ?? 'cash';
    $down_payment = floatval($_POST['down_payment'] ?? 0);

    // Calculate amount to deduct
    $amount_to_deduct = 0;
    $monthly_payment = 0;
    $term_months = 0;

    if ($payment_type === 'cash') {
        $amount_to_deduct = $car_price;
    } elseif (in_array($payment_type, ['lease', 'finance'])) {
        $min_down = $car_price * 0.10;
        $down_payment = max($min_down, $down_payment);
        $amount_to_deduct = $down_payment;
        $term_months = 36;
        $monthly_payment = ($car_price - $down_payment) / $term_months;
    } else {
        echo '<script>alert("Invalid payment method."); window.location="car_details.php?id=' . $car_id . '";</script>';
        exit;
    }

    // Check balance
    if ($balance < $amount_to_deduct) {
        echo '<script>alert("Insufficient balance! You need $' . number_format($amount_to_deduct, 2) . ' but only have $' . number_format($balance, 2) . '."); window.location="car_details.php?id=' . $car_id . '";</script>';
        exit;
    }

    // Start transaction
    $link->autocommit(FALSE);

    try {
        // Insert order
        $order_sql = "INSERT INTO orders (user_id, car_id, amount, status, payment_method, down_payment, monthly_payment, term_months)
                      VALUES (?, ?, ?, 'pending', ?, ?, ?, ?)";
        $order_stmt = $link->prepare($order_sql);
        $order_stmt->bind_param('iidssdi', $user_id, $car_id, $car_price, $payment_type, $down_payment, $monthly_payment, $term_months);
        $order_stmt->execute();
        $order_id = $link->insert_id;
        $order_stmt->close();

        // Insert order item
        $oi_sql = "INSERT INTO order_items (order_id, car_id, quantity, unit_price, total_price) VALUES (?, ?, 1, ?, ?)";
        $oi_stmt = $link->prepare($oi_sql);
        $oi_stmt->bind_param('iidd', $order_id, $car_id, $car_price, $car_price);
        $oi_stmt->execute();
        $oi_stmt->close();

        // Update user balance
        $new_balance = $balance - $amount_to_deduct;
        $upd_sql = "UPDATE users SET balance = ? WHERE id = ?";
        $upd_stmt = $link->prepare($upd_sql);
        $upd_stmt->bind_param('di', $new_balance, $user_id);
        $upd_stmt->execute();
        $upd_stmt->close();


        // Mark car as sold (cash only)
        if ($payment_type === 'cash') {
            $sold_sql = "UPDATE cars SET status = 'sold_out' WHERE id = ?";
            $sold_stmt = $link->prepare($sold_sql);
            $sold_stmt->bind_param('i', $car_id);
            $sold_stmt->execute();
            $sold_stmt->close();
            $car['status'] = 'sold_out'; // Update local copy
        }

        // Commit
        $link->commit();
        $link->autocommit(TRUE);
        
        
        // Success alert
        $msg = "Order #$order_id placed successfully!\\n";
        $msg .= $payment_type === 'cash' ? "Paid in full." : "Down payment: \${$down_payment}";
        echo "<script>alert('$msg'); window.location='car_details.php?id=$car_id';</script>";
        exit;

    } catch (Exception $e) {
        $link->rollback();
        $link->autocommit(TRUE);
        echo '<script>alert("Order failed. Please try again."); window.location="history.php";</script>';
        exit;
    }
}

// --------------------------------------------------
// 5. Helper
// --------------------------------------------------
function fmt($n) { return number_format((float)$n, 2); }
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title><?= htmlspecialchars($car['title']) ?> – Tesla Investment</title>
    <link rel="icon" href="storage/settings/nD0jqatt0igD7oF7VC4CdsckD2tbSAV47lXUHebb.png">
    <link href="https://fonts.bunny.net/css?family=inter:300,400,500,600,700&display=swap" rel="stylesheet"/>
    <script src="https://unpkg.com/lucide@latest/dist/umd/lucide.js"></script>
    <link rel="stylesheet" href="css/app-BrT5j27K.css"/>
</head>
<body class="font-sans antialiased bg-white text-black">

 <!-- Top Navigation (Dashboard-style) -->
    <nav class="fixed top-0 left-0 right-0 z-50 bg-white border-b border-gray-200">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between items-center h-16">
                <!-- Logo -->
                <div class="flex items-center">
                    <a href="dashboard.php" class="flex items-center">
                        <img src="../images/logo.png"
                            alt="Tesla Investment" class="h-6 w-auto">
                    </a>
                </div>

                <!-- Primary Nav -->
                <div class="hidden md:flex items-center space-x-6">
                    <a href="cars.php"
                        class="text-sm font-medium text-gray-700 hover:text-black">Inventory</a>
                    <a href="investments.php"
                        class="text-sm font-medium text-gray-700 hover:text-black">Invest</a>
                    <a href="stocks.php"
                        class="text-sm font-medium text-gray-700 hover:text-black">Stocks</a>
                    <a href="portfolio.php"
                        class="text-sm font-medium text-gray-700 hover:text-black">Portfolio</a>
                </div>

                <!-- Right Side -->
                <div class="flex items-center space-x-2">
                    <!-- Account Dropdown -->
                    <div class="relative group">
                        <button
                            class="flex items-center text-sm font-medium text-gray-700 hover:text-black px-3 py-2 rounded-md">
                            <i data-lucide="user" class="w-4 h-4 mr-2"></i>
                            Account
                        </button>
                        <div
                            class="absolute right-0 mt-2 w-56 bg-white rounded-lg shadow-lg border border-gray-200 opacity-0 invisible group-hover:opacity-100 group-hover:visible transition-all duration-200">
                            <div class="py-2">
                                <div class="px-4 py-3 border-b border-gray-100">
                                    <p class="text-sm font-medium text-black"><?php echo $fullname ?></p>
                                    <p class="text-xs text-gray-500"><?php echo $email ?></p>
                                </div>
                                <a href="dashboard.php"
                                    class="block px-4 py-2 text-sm text-gray-700 hover:bg-gray-50">Dashboard</a>
                                <a href="profile.php"
                                    class="block px-4 py-2 text-sm text-gray-700 hover:bg-gray-50">Profile</a>
                                <div class="border-t border-gray-100 mt-2 pt-2">
                                    <form method="POST" action="logout.php">
                                        <input type="hidden" name="_token"
                                            value="K9QUwxOWNoNg1ohfn8DLAqw9hBayDNimfGmO7gLK" autocomplete="off"> <button
                                            type="submit"
                                            class="block w-full text-left px-4 py-2 text-sm text-gray-700 hover:bg-gray-50">Sign
                                            Out</button>
                                    </form>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Mobile menu button -->
                    <button type="button"
                        class="md:hidden p-2 text-gray-600 hover:text-black rounded-md hover:bg-gray-100"
                        id="mobile-menu-button">
                        <span class="sr-only">Open menu</span>
                        <i data-lucide="menu" class="w-5 h-5" id="menu-icon"></i>
                        <i data-lucide="x" class="w-5 h-5 hidden" id="close-icon"></i>
                    </button>
                </div>
            </div>
        </div>

        <!-- Mobile Menu -->
        <div class="md:hidden hidden bg-white border-t border-gray-200" id="mobile-menu">
            <div class="px-4 pt-2 pb-4 space-y-1">
                <a href="cars"
                    class="block px-3 py-2 text-sm font-medium text-gray-700 hover:bg-gray-50 rounded">Inventory</a>
                <a href="investments.php"
                    class="block px-3 py-2 text-sm font-medium text-gray-700 hover:bg-gray-50 rounded">Invest</a>
                <a href="stocks.php"
                    class="block px-3 py-2 text-sm font-medium text-gray-700 hover:bg-gray-50 rounded">Stocks</a>
                <a href="portfolio.php"
                    class="block px-3 py-2 text-sm font-medium text-gray-700 hover:bg-gray-50 rounded">Portfolio</a>
                <div class="border-t border-gray-200 mt-2 pt-2">
                    <a href="dashboard.php"
                        class="block px-3 py-2 text-sm font-medium text-gray-700 hover:bg-gray-50 rounded">Dashboard</a>
                    <a href="profile.php"
                        class="block px-3 py-2 text-sm font-medium text-gray-700 hover:bg-gray-50 rounded">Profile</a>
                    <form method="POST" action="logout.php" class="mt-1">
                        <input type="hidden" name="_token" value="K9QUwxOWNoNg1ohfn8DLAqw9hBayDNimfGmO7gLK"
                            autocomplete="off"> <button type="submit"
                            class="block w-full text-left px-3 py-2 text-sm font-medium text-gray-700 hover:bg-gray-50 rounded">Sign
                            Out</button>
                    </form>
                </div>
            </div>
        </div>
    </nav>

    <!-- Flash Messages (Dashboard-style) -->
    <div class="pt-16">
    </div>


<main class="min-h-screen bg-white">

    <!-- Hero -->
    <header class="relative overflow-hidden bg-gradient-to-br from-black via-gray-900 to-black text-white">
        <svg class="absolute inset-0 opacity-[0.06] pointer-events-none" xmlns="http://www.w3.org/2000/svg" preserveAspectRatio="none">
            <defs><pattern id="grid" width="40" height="40" patternUnits="userSpaceOnUse">
                <path d="M 40 0 L 0 0 0 40" fill="none" stroke="white" stroke-width="1"/>
            </pattern></defs>
            <rect width="100%" height="100%" fill="url(#grid)"/>
        </svg>
        <div class="absolute inset-0 pointer-events-none">
            <img src="<?= htmlspecialchars($car['image_url']) ?>" alt="" class="h-full w-full object-cover opacity-20">
            <div class="absolute inset-0 bg-gradient-to-b from-black/80 via-black/70 to-black/90"></div>
        </div>
        <div class="relative z-10">
            <div class="mx-auto max-w-7xl px-4 sm:px-6 lg:px-8 py-16 lg:py-24 text-center">
                <h1 class="text-4xl font-semibold tracking-tight sm:text-5xl"><?= htmlspecialchars($car['title']) ?></h1>
                <p class="mt-4 max-w-2xl mx-auto text-base text-gray-300 sm:text-lg">
                    <?= $car['year'] ?> <?= htmlspecialchars($car['model']) ?> – <?= $car['range_miles'] ?> mi range
                </p>

                <div class="mt-10 grid grid-cols-1 gap-3 sm:grid-cols-3">
                    <div class="rounded-lg border border-white/10 bg-white/5 p-4">
                        <p class="text-xs text-gray-300">Range</p>
                        <p class="mt-1 text-2xl font-light text-white"><?= $car['range_miles'] ?> <span class="text-lg">mi</span></p>
                    </div>
                    <div class="rounded-lg border border-white/10 bg-white/5 p-4">
                        <p class="text-xs text-gray-300">0-60 mph</p>
                        <p class="mt-1 text-2xl font-light text-white"><?= $car['accel_0_60'] ?> <span class="text-lg">s</span></p>
                    </div>
                    <div class="rounded-lg border border-white/10 bg-white/5 p-4">
                        <p class="text-xs text-gray-300">Top Speed</p>
                        <p class="mt-1 text-2xl font-light text-white"><?= $car['top_speed'] ?> <span class="text-lg">mph</span></p>
                    </div>
                </div>

                <?php if ($car['status'] === 'available'): ?>
                    <div class="mt-8">
                        <a href="#order" class="inline-flex items-center rounded-md bg-white px-8 py-3 text-sm font-medium text-black hover:bg-gray-100">Order Now</a>
                    </div>
                <?php else: ?>
                    <div class="mt-8">
                        <span class="inline-flex items-center rounded-md bg-red-600 px-8 py-3 text-sm font-medium text-white">Sold Out</span>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        <div class="absolute bottom-0 left-0 right-0 translate-y-[1px]">
            <svg viewBox="0 0 1440 80" xmlns="http://www.w3.org/2000/svg">
                <path fill="#ffffff" d="M0,64L48,69.3C96,75,192,85,288,85.3C384,85,480,75,576,53.3C672,32,768,0,864,0C960,0,1056,32,1152,53.3C1248,75,1344,85,1392,90.7L1440,96L1440,160L0,160Z"></path>
            </svg>
        </div>
    </header>

    <!-- Order Section -->
    <section id="order" class="py-16 bg-white">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <?php if ($car['status'] !== 'available'): ?>
                <div class="text-center py-12">
                    <h2 class="text-2xl font-medium text-red-600">This Vehicle is Sold Out</h2>
                    <p class="text-gray-600 mt-2">It has already been purchased.</p>
                    <a href="cars.php" class="mt-4 inline-block bg-black text-white px-6 py-3 rounded-lg hover:bg-gray-800">Browse Available Cars</a>
                </div>
            <?php else: ?>
                <div class="grid grid-cols-1 lg:grid-cols-2 gap-12">
                    <!-- Image -->
                    <div>
                        <img id="main-img" src="<?= htmlspecialchars($car['image_url']) ?>" alt="<?= htmlspecialchars($car['title']) ?>" class="w-full rounded-xl border border-gray-200">
                        <div class="grid grid-cols-4 gap-2 mt-4">
                            <?php for($i=0; $i<4; $i++): ?>
                                <button onclick="document.getElementById('main-img').src='<?= htmlspecialchars($car['image_url']) ?>'" class="border-2 <?= $i===0?'border-black':'border-gray-300' ?> rounded-lg overflow-hidden">
                                    <img src="<?= htmlspecialchars($car['image_url']) ?>" class="w-full h-16 object-cover">
                                </button>
                            <?php endfor; ?>
                        </div>
                    </div>

                    <!-- Order Form -->
                    <div>
                        <div class="border border-gray-200 rounded-2xl p-6 shadow-sm">
                            <h2 class="text-3xl font-light mb-2">Order Your <?= htmlspecialchars($car['title']) ?></h2>
                            <p class="text-gray-600 mb-6">Balance: <strong>$<?= fmt($balance) ?></strong></p>

                            <form method="POST" id="order-form">
                                <input type="hidden" name="place_order" value="1">

                                <!-- Payment Tabs -->
                                <div class="flex border-b mb-6">
                                    <button type="button" class="tab flex-1 py-3 text-sm font-medium" data-type="cash">Purchase</button>
                                    <button type="button" class="tab flex-1 py-3 text-sm font-medium text-black border-b-2 border-black" data-type="lease">Lease</button>
                                    <button type="button" class="tab flex-1 py-3 text-sm font-medium" data-type="finance">Finance</button>
                                </div>

                                <input type="hidden" name="payment_type" id="payment_type" value="lease">

                                <!-- Price -->
                                <div class="text-center mb-6">
                                    <div class="text-4xl font-light" id="price-display">$<?= fmt($car['price']) ?></div>
                                    <p class="text-sm text-gray-500" id="price-details">Est. Purchase Price</p>
                                </div>

                                <!-- Down Payment -->
                                <div id="down-payment-section" class="mb-6">
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Down Payment</label>
                                    <input type="number" name="down_payment" id="down_payment" step="0.01" min="<?= $car['price']*0.1 ?>" 
                                           value="<?= fmt($car['price']*0.1) ?>" class="w-full px-3 py-2 border rounded-lg">
                                    <p class="text-xs text-gray-500 mt-1">Minimum 10% ($<?= fmt($car['price']*0.1) ?>)</p>
                                </div>

                                <!-- Gas Savings -->
                                <div class="flex items-center mb-6">
                                    <input type="checkbox" id="gas-savings" class="w-4 h-4">
                                    <label for="gas-savings" class="ml-2 text-sm text-gray-600">Include est. gas savings ($108/mo)</label>
                                </div>

                                <!-- Submit -->
                                <button type="submit" id="order-btn" 
                                        class="w-full bg-black text-white py-4 rounded-lg font-medium hover:bg-gray-800 disabled:bg-gray-400 disabled:cursor-not-allowed">
                                    Complete Purchase
                                </button>
                            </form>
                        </div>
                    </div>
                </div>
            <?php endif; ?>
        </div>
    </section>

    <!-- Specs -->
    <section class="py-16 bg-gray-50">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <h2 class="text-3xl font-light text-center mb-12">Specifications</h2>
            <div class="grid md:grid-cols-2 gap-12 max-w-4xl mx-auto">
                <div>
                    <h3 class="text-xl font-medium mb-6">Performance</h3>
                    <div class="space-y-4 text-sm">
                        <div class="flex justify-between border-b pb-2"><span class="text-gray-600">0-60 mph</span><span><?= $car['accel_0_60'] ?> s</span></div>
                        <div class="flex justify-between border-b pb-2"><span class="text-gray-600">Top Speed</span><span><?= $car['top_speed'] ?> mph</span></div>
                        <div class="flex justify-between border-b pb-2"><span class="text-gray-600">Range</span><span><?= $car['range_miles'] ?> mi</span></div>
                    </div>
                </div>
                <div>
                    <h3 class="text-xl font-medium mb-6">Details</h3>
                    <div class="space-y-4 text-sm">
                        <div class="flex justify-between border-b pb-2"><span class="text-gray-600">Model</span><span><?= htmlspecialchars($car['model']) ?></span></div>
                        <div class="flex justify-between border-b pb-2"><span class="text-gray-600">Year</span><span><?= $car['year'] ?></span></div>
                        <div class="flex justify-between border-b pb-2"><span class="text-gray-600">Price</span><span>$<?= fmt($car['price']) ?></span></div>
                        <div class="flex justify-between border-b pb-2"><span class="text-gray-600">Status</span><span class="<?= $car['status']==='available'?'text-green-600':'text-red-600' ?> font-medium"><?= ucfirst($car['status']) ?></span></div>
                    </div>
                </div>
            </div>
        </div>
    </section>
</main>

<?php include 'footer.php'; ?>

<script>
document.addEventListener('DOMContentLoaded', () => {
    const price = <?= $car['price'] ?>;
    const balance = <?= $balance ?>;
    let type = 'lease';
    const gas = 108;

    const display = document.getElementById('price-display');
    const details = document.getElementById('price-details');
    const downSection = document.getElementById('down-payment-section');
    const downInput = document.getElementById('down_payment');
    const typeInput = document.getElementById('payment_type');
    const btn = document.getElementById('order-btn');
    const gasChk = document.getElementById('gas-savings');

    function update() {
        let show = price;
        let txt = 'Est. Purchase Price';
        downSection.style.display = 'none';
        btn.disabled = balance < price;

        if (type !== 'cash') {
            const min = price * 0.1;
            const down = Math.max(min, parseFloat(downInput.value) || min);
            downInput.value = down.toFixed(2);
            show = down;
            txt = type === 'lease' ? 'Down Payment (36 mo)' : 'Down Payment (Finance)';
            downSection.style.display = 'block';
            btn.disabled = balance < down;
        }

        if (gasChk.checked && type !== 'cash') show = Math.max(0, show - gas);

        display.textContent = '$' + show.toLocaleString();
        details.textContent = txt;
    }

    document.querySelectorAll('.tab').forEach(t => {
        t.addEventListener('click', () => {
            document.querySelectorAll('.tab').forEach(x => {
                x.classList.remove('text-black', 'border-b-2', 'border-black');
                x.classList.add('text-gray-500');
            });
            t.classList.add('text-black', 'border-b-2', 'border-black');
            t.classList.remove('text-gray-500');
            type = t.dataset.type;
            typeInput.value = type;
            update();
        });
    });

    downInput?.addEventListener('input', update);
    gasChk?.addEventListener('change', update);
    update();
});
</script>

<?php $link->close(); ?>
</body>
</html>